<?php
/**
 * Plugin Name: WPアソシエイト カードヘルパー
 * Plugin URI: https://sui-lab.info/archives/5799
 * Description: WPアソシエイトポストR2からの移行とAmazon商品カードの作成・管理を支援します。
 * Version: 2.0.0
 * Author: SUILAB
 * Author URI: https://sui-lab.info/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: wpap-card-helper
 * Domain Path: /languages
 */

// 直接アクセス禁止
if (!defined('ABSPATH')) {
    exit;
}

// プラグイン定数
define('WPAP_CARD_HELPER_VERSION', '2.0.0');
define('WPAP_CARD_HELPER_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('WPAP_CARD_HELPER_PLUGIN_URL', plugin_dir_url(__FILE__));
define('WPAP_CARD_HELPER_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * メインプラグインクラス
 */
class WPAP_Card_Helper {
    
    /**
     * シングルトンインスタンス
     */
    private static $instance = null;
    
    /**
     * 設定クラス
     */
    public $settings;
    
    /**
     * Amazon APIクラス
     */
    public $amazon_api;
    
    /**
     * ショートコードプロセッサクラス
     */
    public $shortcode_processor;
    
    /**
     * カード生成クラス
     */
    public $card_generator;
    
    /**
     * REST APIクラス
     */
    public $rest_api;
    
    /**
     * 新しいショートコードクラス
     */
    public $new_shortcode;
    
    /**
     * シングルトンインスタンスを取得
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * コンストラクタ
     */
    private function __construct() {
        $this->load_dependencies();
        $this->init_classes();
        $this->init_hooks();
    }
    
    /**
     * 依存ファイルの読み込み
     */
    private function load_dependencies() {
        require_once WPAP_CARD_HELPER_PLUGIN_DIR . 'includes/class-settings.php';
        require_once WPAP_CARD_HELPER_PLUGIN_DIR . 'includes/class-amazon-api.php';
        require_once WPAP_CARD_HELPER_PLUGIN_DIR . 'includes/class-shortcode-processor.php';
        require_once WPAP_CARD_HELPER_PLUGIN_DIR . 'includes/class-card-generator.php';
        require_once WPAP_CARD_HELPER_PLUGIN_DIR . 'includes/class-rest-api.php';
        require_once WPAP_CARD_HELPER_PLUGIN_DIR . 'includes/class-new-shortcode.php';
        require_once WPAP_CARD_HELPER_PLUGIN_DIR . 'includes/class-bulk-processor.php';
        require_once WPAP_CARD_HELPER_PLUGIN_DIR . 'includes/class-editor-integration.php';
    }
    
    /**
     * クラスの初期化
     */
    private function init_classes() {
        $this->settings = new WPAP_Card_Helper_Settings();
        $this->amazon_api = new WPAP_Card_Helper_Amazon_API();
        $this->shortcode_processor = new WPAP_Card_Helper_Shortcode_Processor();
        $this->card_generator = new WPAP_Card_Helper_Card_Generator();
        $this->rest_api = new WPAP_Card_Helper_REST_API();
        $this->new_shortcode = new WPAP_Card_Helper_New_Shortcode();
        new WPAP_Card_Helper_Bulk_Processor();
        new WPAP_Card_Helper_Editor_Integration();
    }
    
    /**
     * フックの初期化
     */
    private function init_hooks() {
        // 管理画面スクリプト・スタイル
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        
        // フロントエンドスタイル
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_assets'));
        
        // クラシックエディタ用メタボックス
        add_action('add_meta_boxes', array($this, 'add_meta_box'));
        
        // プラグインリンク
        add_filter('plugin_action_links_' . WPAP_CARD_HELPER_PLUGIN_BASENAME, array($this, 'add_plugin_links'));
    }
    
    /**
     * 管理画面アセットの読み込み
     */
    public function enqueue_admin_assets($hook) {
        global $post;
        
        // 設定ページ
        if ('settings_page_wpap-card-helper' === $hook) {
            wp_enqueue_style(
                'wpap-card-helper-admin',
                WPAP_CARD_HELPER_PLUGIN_URL . 'assets/css/admin-style.css',
                array(),
                WPAP_CARD_HELPER_VERSION
            );
            return;
        }
        
        // 投稿編集画面
        if ('post.php' === $hook || 'post-new.php' === $hook) {
            wp_enqueue_style(
                'wpap-card-helper-admin',
                WPAP_CARD_HELPER_PLUGIN_URL . 'assets/css/admin-style.css',
                array(),
                WPAP_CARD_HELPER_VERSION
            );
            
            wp_enqueue_style(
                'wpap-card-helper-card',
                WPAP_CARD_HELPER_PLUGIN_URL . 'assets/css/card-style.css',
                array(),
                WPAP_CARD_HELPER_VERSION
            );
            
            wp_enqueue_script(
                'wpap-card-helper-editor',
                WPAP_CARD_HELPER_PLUGIN_URL . 'assets/js/classic-editor.js',
                array('jquery'),
                WPAP_CARD_HELPER_VERSION,
                true
            );
            
            wp_localize_script('wpap-card-helper-editor', 'wpapCardHelperData', array(
                'restUrl' => rest_url('wpap-card-helper/v1/'),
                'nonce' => wp_create_nonce('wp_rest'),
                'postId' => $post ? $post->ID : 0,
                'ajaxUrl' => admin_url('admin-ajax.php'),
            ));
        }
    }
    
    /**
     * フロントエンドアセットの読み込み
     */
    public function enqueue_frontend_assets() {
        wp_enqueue_style(
            'wpap-card-helper-card',
            WPAP_CARD_HELPER_PLUGIN_URL . 'assets/css/card-style.css',
            array(),
            WPAP_CARD_HELPER_VERSION
        );
    }
    
    /**
     * クラシックエディタ用メタボックスの追加
     */
    public function add_meta_box() {
        $post_types = array('post', 'page');
        
        foreach ($post_types as $post_type) {
            add_meta_box(
                'wpap-card-helper-metabox',
                'WPAP カードヘルパー',
                array($this, 'render_meta_box'),
                $post_type,
                'side',
                'high'
            );
        }
    }
    
    /**
     * メタボックスの描画
     */
    public function render_meta_box($post) {
        wp_nonce_field('wpap_card_helper_metabox', 'wpap_card_helper_nonce');
        ?>
        <div id="wpap-card-helper-panel">
            <p class="description">WPアソシエイトポストR2のショートコードを検出・置換します。</p>
            
            <div id="wpap-card-helper-status" style="margin: 10px 0; padding: 10px; background: #f0f0f0; border-radius: 4px; display: none;">
                <span id="wpap-status-text"></span>
            </div>
            
            <div id="wpap-card-helper-results" style="margin: 10px 0; max-height: 200px; overflow-y: auto;">
                <!-- 検出結果がここに表示される -->
            </div>
            
            <div class="wpap-card-helper-buttons" style="display: flex; flex-direction: column; gap: 8px;">
                <button type="button" id="wpap-scan-btn" class="button button-primary" style="width: 100%;">
                    🔍 ショートコードを検出
                </button>
                <button type="button" id="wpap-replace-btn" class="button" style="width: 100%;" disabled>
                    ✨ すべて置換
                </button>
                <button type="button" id="wpap-revert-btn" class="button" style="width: 100%;" disabled>
                    ↩️ 元に戻す
                </button>
            </div>
            
            <div id="wpap-card-helper-progress" style="margin-top: 10px; display: none;">
                <div style="background: #ddd; border-radius: 4px; overflow: hidden;">
                    <div id="wpap-progress-bar" style="width: 0%; height: 20px; background: #0073aa; transition: width 0.3s;"></div>
                </div>
                <p id="wpap-progress-text" style="text-align: center; margin: 5px 0; font-size: 12px;"></p>
            </div>
        </div>
        <?php
    }
    
    /**
     * プラグインリンクの追加
     */
    public function add_plugin_links($links) {
        $settings_link = '<a href="' . admin_url('options-general.php?page=wpap-card-helper') . '">' . __('設定', 'wpap-card-helper') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
}

/**
 * プラグインのアクティベーション
 */
function wpap_card_helper_activate() {
    // デフォルトオプションの設定
    if (false === get_option('wpap_replacer_settings')) {
        add_option('wpap_replacer_settings', array(
            'credential_id' => '',
            'credential_secret' => '',
            'tracking_id' => '',
        ));
    }
    
    // キャッシュ用トランジェントのクリア
    delete_transient('wpap_replacer_token');
}
register_activation_hook(__FILE__, 'wpap_card_helper_activate');

/**
 * プラグインの非アクティベーション
 */
function wpap_card_helper_deactivate() {
    // トランジェントのクリア
    delete_transient('wpap_replacer_token');
}
register_deactivation_hook(__FILE__, 'wpap_card_helper_deactivate');

/**
 * プラグインの初期化
 */
function wpap_card_helper_init() {
    return WPAP_Card_Helper::get_instance();
}
add_action('plugins_loaded', 'wpap_card_helper_init');
