<?php
/**
 * ショートコード処理クラス
 */

if (!defined('ABSPATH')) {
    exit;
}

class WPAP_Card_Helper_Shortcode_Processor {
    
    /**
     * WPAPショートコードの正規表現パターン
     */
    const SHORTCODE_PATTERN = '/\[wpap\s+([^\]]+)\]/';
    
    /**
     * 元のショートコードを保持するHTMLコメントパターン
     */
    const ORIGINAL_COMMENT_PATTERN = '/<!--\s*wpap-original:\s*(\[wpap[^\]]+\])\s*-->/';
    
    /**
     * 置換済みカードのパターン（HTMLカード形式）
     * 改行を含む任意の文字列にマッチするよう [\s\S] を使用
     */
    const REPLACED_CARD_PATTERN = '/<!--\s*wpap-original:\s*\[wpap[^\]]+\]\s*-->[\s\S]*?<div class="wpap-replacer-card"[^>]*>[\s\S]*?<\/div>[\s\S]*?<!--\s*\/wpap-replacer\s*-->/';
    
    /**
     * 置換済みショートコードのパターン（新ショートコード形式）
     * 閉じるコメント <!-- /wpap-replacer --> も含む
     * 注: 次のwpap-originalが出るまでの範囲でマッチするよう制限
     */
    const REPLACED_SHORTCODE_PATTERN = '/<!--\s*wpap-original:\s*(\[wpap[^\]]+\])\s*-->\s*\[wpap_card[^\]]*\]\s*<!--\s*\/wpap-replacer\s*-->/';
    
    /**
     * コンストラクタ
     */
    public function __construct() {
        // 必要に応じて初期化処理
    }
    
    /**
     * コンテンツ内のWPAPショートコードを検出
     * 
     * @param string $content 投稿コンテンツ
     * @return array 検出されたショートコード情報の配列
     */
    public function scan_shortcodes($content) {
        $shortcodes = array();
        
        // ========================================
        // Step 0: ネストされた重複パターンを修復
        // ========================================
        // パターン: <!-- wpap-original: <!-- wpap-original: [wpap...] --> ... --> を修復
        $content = $this->fix_nested_replacements($content);
        
        // ========================================
        // Step 1: 置換済みショートコードの検出
        // ========================================
        // 定数のパターンを使用して置換済みを検出
        // REPLACED_SHORTCODE_PATTERN: <!-- wpap-original: ([wpap...]) --> [wpap_card...] <!-- /wpap-replacer -->
        
        // まず、置換済みブロックを検出（HTMLカード形式）
        if (preg_match_all(self::REPLACED_CARD_PATTERN, $content, $card_matches, PREG_SET_ORDER)) {
            foreach ($card_matches as $match) {
                // コメントから元のショートコードを抽出
                if (preg_match(self::ORIGINAL_COMMENT_PATTERN, $match[0], $original)) {
                    $original_sc = $original[1];
                    if (preg_match(self::SHORTCODE_PATTERN, $original_sc, $sc_match)) {
                        $parsed = $this->parse_shortcode_attributes($sc_match[1]);
                        if (!empty($parsed['id'])) {
                            $shortcodes[] = $this->create_shortcode_info($original_sc, $parsed, true);
                        }
                    }
                }
            }
        }
        
        // 次に、置換済みブロックを検出（新ショートコード形式）
        if (preg_match_all(self::REPLACED_SHORTCODE_PATTERN, $content, $sc_block_matches, PREG_SET_ORDER)) {
            foreach ($sc_block_matches as $match) {
                $original_sc = $match[1]; // キャプチャグループから元のショートコード
                if (preg_match(self::SHORTCODE_PATTERN, $original_sc, $sc_match)) {
                    $parsed = $this->parse_shortcode_attributes($sc_match[1]);
                    if (!empty($parsed['id'])) {
                        $shortcodes[] = $this->create_shortcode_info($original_sc, $parsed, true);
                    }
                }
            }
        }
        
        // ========================================
        // Step 2: 未置換ショートコードの検出
        // ========================================
        // 置換済みブロックを除去してから検索
        $content_clean = $content;
        $content_clean = preg_replace(self::REPLACED_CARD_PATTERN, '', $content_clean);
        $content_clean = preg_replace(self::REPLACED_SHORTCODE_PATTERN, '', $content_clean);
        
        // 残ったコンテンツから [wpap ...] を検索（これが未置換）
        if (preg_match_all(self::SHORTCODE_PATTERN, $content_clean, $matches, PREG_SET_ORDER)) {
            foreach ($matches as $match) {
                $full_shortcode = $match[0];
                $attributes_string = $match[1];
                
                $parsed = $this->parse_shortcode_attributes($attributes_string);
                
                if (!empty($parsed['id'])) {
                    $shortcodes[] = $this->create_shortcode_info($full_shortcode, $parsed, false);
                }
            }
        }
        
        return $shortcodes;
    }
    
    /**
     * ショートコード情報配列を生成
     * 
     * @param string $shortcode ショートコード文字列
     * @param array $parsed パース済み属性
     * @param bool $is_replaced 置換済みフラグ
     * @return array ショートコード情報
     */
    private function create_shortcode_info($shortcode, $parsed, $is_replaced) {
        return array(
            'shortcode' => $shortcode,
            'position' => 0,
            'asin' => $parsed['id'],
            'title' => $parsed['title'] ?? '',
            'search' => $parsed['search'] ?? '',
            'service' => $parsed['service'] ?? 'with',
            'type' => $parsed['type'] ?? 'detail',
            'is_replaced' => $is_replaced,
        );
    }
    
    /**
     * ショートコード属性のパース
     * 
     * @param string $attributes_string 属性文字列
     * @return array パースされた属性の連想配列
     */
    private function parse_shortcode_attributes($attributes_string) {
        $attributes = array();
        
        // 正規表現で属性を抽出
        $pattern = '/(\w+)=(["\'])([^"\']*)\2/';
        
        if (preg_match_all($pattern, $attributes_string, $matches, PREG_SET_ORDER)) {
            foreach ($matches as $match) {
                $attributes[$match[1]] = $match[3];
            }
        }
        
        return $attributes;
    }
    
    /**
     * ショートコードをカードHTMLに置換
     * 
     * @param string $content 投稿コンテンツ
     * @param string $shortcode 置換対象のショートコード
     * @param string $card_html 置換後のカードHTML
     * @return string 置換後のコンテンツ
     */
    public function replace_shortcode($content, $shortcode, $card_html) {
        // ショートコードをエスケープしてパターンを作成
        $escaped_shortcode = preg_quote($shortcode, '/');
        
        // 既にコメント内にあるショートコードは置換しない
        // 否定先読み・後読みを使用：<!-- wpap-original: の後にある場合はスキップ
        // パターン：コメント内ではない[wpap...]にのみマッチ
        $pattern = '/(?<!wpap-original: )' . $escaped_shortcode . '(?!\s*-->)/';
        
        // 元のショートコードをHTMLコメントとして保持しつつカードに置換
        $replacement = '<!-- wpap-original: ' . $shortcode . ' -->' . "\n" . $card_html . "\n" . '<!-- /wpap-replacer -->';
        
        return preg_replace($pattern, $replacement, $content, 1);
    }
    
    /**
     * カードを元のショートコードに復元
     * 
     * @param string $content 投稿コンテンツ
     * @return string 復元後のコンテンツ
     */
    public function revert_all($content) {
        // まず破損したコンテンツを修復
        $content = $this->repair_corrupted_content($content);
        
        // HTMLカード形式を復元
        $content = preg_replace_callback(
            self::REPLACED_CARD_PATTERN,
            function($matches) {
                // コメントから元のショートコードを抽出
                if (preg_match(self::ORIGINAL_COMMENT_PATTERN, $matches[0], $original)) {
                    return $original[1];
                }
                return $matches[0];
            },
            $content
        );
        
        // 新ショートコード形式（wpap_card）を復元
        $content = preg_replace_callback(
            self::REPLACED_SHORTCODE_PATTERN,
            function($matches) {
                // キャプチャグループから元のショートコードを取得
                return $matches[1];
            },
            $content
        );
        
        return $content;
    }
    
    /**
     * 特定のASINのカードを元に戻す
     * 
     * @param string $content 投稿コンテンツ
     * @param string $asin 復元対象のASIN
     * @return string 復元後のコンテンツ
     */
    public function revert_by_asin($content, $asin) {
        $escaped_asin = preg_quote($asin, '/');
        
        // HTMLカード形式を復元
        $html_pattern = '/<!-- wpap-original: (\[wpap[^]]*id=["\']' . $escaped_asin . '["\'][^\]]*\]) -->\s*<div class="wpap-replacer-card"[^>]*>.*?<\/div>\s*<!-- \/wpap-replacer -->/s';
        $content = preg_replace_callback(
            $html_pattern,
            function($matches) {
                return $matches[1];
            },
            $content
        );
        
        // 新ショートコード形式を復元
        $shortcode_pattern = '/<!-- wpap-original: (\[wpap[^]]*id=["\']' . $escaped_asin . '["\'][^\]]*\]) -->\s*\[wpap_card[^\]]*\]\s*<!-- \/wpap-replacer -->/s';
        $content = preg_replace_callback(
            $shortcode_pattern,
            function($matches) {
                return $matches[1];
            },
            $content
        );
        
        return $content;
    }
    
    /**
     * コンテンツに復元可能なカードがあるかチェック
     * 
     * @param string $content 投稿コンテンツ
     * @return bool
     */
    public function has_replaced_cards($content) {
        return preg_match(self::ORIGINAL_COMMENT_PATTERN, $content) === 1;
    }
    
    /**
     * コンテンツに未置換のショートコードがあるかチェック
     * 
     * @param string $content 投稿コンテンツ
     * @return bool
     */
    public function has_unreplaced_shortcodes($content) {
        // まず全てのショートコードを見つける
        if (!preg_match_all(self::SHORTCODE_PATTERN, $content, $matches, PREG_OFFSET_CAPTURE)) {
            return false;
        }
        
        // 各ショートコードが既に置換済みかチェック
        foreach ($matches[0] as $match) {
            $position = $match[1];
            $shortcode = $match[0];
            
            // このショートコードの前にHTMLコメントがあるかチェック
            $before_content = substr($content, max(0, $position - 50), 50);
            if (strpos($before_content, '<!-- wpap-original:') === false) {
                return true; // 未置換のショートコードが見つかった
            }
        }
        
        return false;
    }
    
    /**
     * 破損したコンテンツを修復
     * ネストされた<!-- wpap-original: -->コメントを正常な形式に戻す
     * 
     * @param string $content 投稿コンテンツ
     * @return string 修復後のコンテンツ
     */
    public function repair_corrupted_content($content) {
        // ネストされたwpap-originalコメントを検出して修復
        // パターン: <!-- wpap-original: <!-- wpap-original: ... --> --> を検出
        $max_iterations = 10; // 無限ループ防止
        $iteration = 0;
        
        while ($iteration < $max_iterations) {
            $iteration++;
            
            // ネストされたコメントのパターン
            $pattern = '/<!-- wpap-original: <!-- wpap-original:/';
            if (!preg_match($pattern, $content)) {
                break; // ネストがなくなったら終了
            }
            
            // 最も内側のwpap-originalを見つけて、外側を削除
            // <!-- wpap-original: <!-- wpap-original: [wpap ...] --> を
            // <!-- wpap-original: [wpap ...] --> に修正
            $content = preg_replace(
                '/<!-- wpap-original: (<!-- wpap-original: \[wpap[^\]]+\] -->)/',
                '$1',
                $content
            );
        }
        
        // 重複した[wpap_card]と<!-- /wpap-replacer -->を削除
        // パターン: [wpap_card ...]\n<!-- /wpap-replacer --> -->\n[wpap_card ...] のような重複
        $content = preg_replace(
            '/(\[wpap_card[^\]]+\])\s*<!-- \/wpap-replacer -->\s*-->\s*\[wpap_card[^\]]+\]\s*<!-- \/wpap-replacer -->\s*-->\s*\[wpap_card[^\]]+\]\s*<!-- \/wpap-replacer -->\s*-->\s*(\[wpap_card[^\]]+\]\s*<!-- \/wpap-replacer -->)/',
            '$2',
            $content
        );
        
        // より単純なパターンの重複も削除
        $content = preg_replace(
            '/(\[wpap_card[^\]]+\])\s*<!-- \/wpap-replacer -->\s*-->\s*(\[wpap_card[^\]]+\]\s*<!-- \/wpap-replacer -->)/',
            '$2',
            $content
        );
        
        // ネストされた閉じタグを修正
        $content = preg_replace(
            '/<!-- \/wpap-replacer -->\s*-->/',
            '<!-- /wpap-replacer -->',
            $content
        );
        
        // 元のショートコードの直後に残っている孤立したwpap_cardを削除
        // パターン: [wpap ...]\n[wpap_card ...]\n<!-- /wpap-replacer -->
        $content = preg_replace(
            '/(\[wpap\s+[^\]]+\])\s*\[wpap_card[^\]]+\]\s*<!-- \/wpap-replacer -->/',
            '$1',
            $content
        );
        
        // 孤立した[wpap_card]...<!-- /wpap-replacer -->を削除（前に[wpap...]がない場合）
        // これは復元後の残骸
        $content = preg_replace(
            '/(?<!\])\s*\[wpap_card[^\]]+\]\s*<!-- \/wpap-replacer -->/',
            '',
            $content
        );
        
        // 孤立した<!-- wpap-original: [wpap ...] -->コメントを元のショートコードに変換
        // wpap_cardや/wpap-replacerが続かない孤立したコメント
        $content = preg_replace_callback(
            '/<!-- wpap-original: (\[wpap[^\]]+\]) -->(?!\s*\[wpap_card)(?!\s*<div)/',
            function($matches) {
                return $matches[1];
            },
            $content
        );
        
        return $content;
    }
    
    /**
     * ネストされた重複パターンを修復
     * 
     * @param string $content 投稿コンテンツ
     * @return string 修復後のコンテンツ
     */
    public function fix_nested_replacements($content) {
        $max_iterations = 10;
        $iteration = 0;
        
        while ($iteration < $max_iterations) {
            $iteration++;
            $original_content = $content;
            
            // 各修復パターンを順次実行
            $content = $this->fix_nested_comments($content);
            $content = $this->fix_duplicate_closing_tags($content);
            $content = $this->fix_duplicate_card_blocks($content);
            
            // 変更がなければ終了
            if ($content === $original_content) {
                break;
            }
        }
        
        return $content;
    }
    
    /**
     * ネストされたコメント開始を修復
     * 
     * @param string $content コンテンツ
     * @return string 修復後のコンテンツ
     */
    private function fix_nested_comments($content) {
        // <!-- wpap-original: <!-- wpap-original: [wpap...] --> ... --> を
        // <!-- wpap-original: [wpap...] --> ... に修正
        return preg_replace(
            '/<!--\s*wpap-original:\s*<!--\s*wpap-original:\s*(\[wpap[^\]]+\])\s*-->/',
            '<!-- wpap-original: $1 -->',
            $content
        );
    }
    
    /**
     * 重複した閉じタグを修復
     * 
     * @param string $content コンテンツ
     * @return string 修復後のコンテンツ
     */
    private function fix_duplicate_closing_tags($content) {
        // <!-- /wpap-replacer --> --> を <!-- /wpap-replacer --> に修正
        return preg_replace(
            '/<!--\s*\/wpap-replacer\s*-->\s*-->/',
            '<!-- /wpap-replacer -->',
            $content
        );
    }
    
    /**
     * 重複したカードブロックを削除
     * 
     * @param string $content コンテンツ
     * @return string 修復後のコンテンツ
     */
    private function fix_duplicate_card_blocks($content) {
        // 直接連続している重複のみ削除（間に<!-- wpap-original:がない場合）
        return preg_replace(
            '/(\[wpap_card[^\]]+\]\s*<!--\s*\/wpap-replacer\s*-->)\s*(?!<!--\s*wpap-original:)\[wpap_card[^\]]+\]\s*<!--\s*\/wpap-replacer\s*-->/',
            '$1',
            $content
        );
    }
}
