<?php
/**
 * 設定画面クラス
 */

if (!defined('ABSPATH')) {
    exit;
}

class WPAP_Card_Helper_Settings {
    
    /**
     * オプション名
     */
    const OPTION_NAME = 'wpap_replacer_settings';
    
    /**
     * コンストラクタ
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_menu'));
        add_action('admin_init', array($this, 'register_settings'));
    }
    
    /**
     * 管理メニューの追加
     */
    public function add_menu() {
        add_options_page(
            __('WPAP リプレイサー設定', 'wpap-card-helper'),
            __('WPAP リプレイサー', 'wpap-card-helper'),
            'manage_options',
            'wpap-card-helper',
            array($this, 'render_settings_page')
        );
    }
    
    /**
     * 設定の登録
     */
    public function register_settings() {
        register_setting(
            'wpap_replacer_options',
            self::OPTION_NAME,
            array($this, 'sanitize_settings')
        );
        
        // Amazon Creators APIセクション
        add_settings_section(
            'wpap_replacer_api_section',
            __('Amazon Creators API 設定', 'wpap-card-helper'),
            array($this, 'render_api_section'),
            'wpap-card-helper'
        );

        // APIを使用しないオプション
        add_settings_field(
            'disable_api',
            __('APIを使用しない', 'wpap-card-helper'),
            array($this, 'render_disable_api_field'),
            'wpap-card-helper',
            'wpap_replacer_api_section'
        );
        
        // Credential ID
        add_settings_field(
            'credential_id',
            'Credential ID',
            array($this, 'render_credential_id_field'),
            'wpap-card-helper',
            'wpap_replacer_api_section'
        );
        
        // Credential Secret
        add_settings_field(
            'credential_secret',
            'Credential Secret',
            array($this, 'render_credential_secret_field'),
            'wpap-card-helper',
            'wpap_replacer_api_section'
        );
        
        // トラッキングID
        add_settings_field(
            'tracking_id',
            __('アソシエイト トラッキングID', 'wpap-card-helper'),
            array($this, 'render_tracking_id_field'),
            'wpap-card-helper',
            'wpap_replacer_api_section'
        );

        // キャッシュ時間
        add_settings_field(
            'cache_hours',
            __('キャッシュ時間', 'wpap-card-helper'),
            array($this, 'render_cache_hours_field'),
            'wpap-card-helper',
            'wpap_replacer_api_section'
        );
    }
    
    /**
     * API無効化フィールド
     */
    public function render_disable_api_field() {
        $options = get_option(self::OPTION_NAME);
        $checked = isset($options['disable_api']) && $options['disable_api'] ? 'checked' : '';
        echo '<label>';
        echo '<input type="checkbox" name="' . esc_attr(self::OPTION_NAME) . '[disable_api]" value="1" ' . $checked . '>';
        echo ' ' . esc_html__('APIを使用せずにシンプルなリンクカードを生成する', 'wpap-card-helper');
        echo '</label>';
        echo '<p class="description">';
        echo esc_html__('チェックすると、Amazonから商品情報を取得せず、ショートコードのtitle属性を使用したシンプルなカードに置換します。', 'wpap-card-helper') . '<br>';
        echo '<strong>' . esc_html__('画像・価格は表示されません', 'wpap-card-helper') . '</strong>' . esc_html__('が、API資格要件を満たしていない場合でも使用できます。', 'wpap-card-helper');
        echo '</p>';
    }
    
    /**
     * APIセクションの説明
     */
    public function render_api_section() {
        printf(
            '<p>%s<a href="%s" target="_blank">%s</a>%s</p>',
            esc_html__('クレデンシャル情報は', 'wpap-card-helper'),
            esc_url('https://affiliate.amazon.co.jp/'),
            esc_html__('Amazonアソシエイト', 'wpap-card-helper'),
            esc_html__('の管理画面 → ツール → クリエイターAPI から取得できます。', 'wpap-card-helper')
        );
    }
    
    /**
     * キャッシュ時間フィールド
     */
    public function render_cache_hours_field() {
        $options = get_option(self::OPTION_NAME);
        $cache_hours = isset($options['cache_hours']) ? intval($options['cache_hours']) : 24;
        echo '<select name="' . esc_attr(self::OPTION_NAME) . '[cache_hours]">';
        $choices = array(
            1 => __('1時間', 'wpap-card-helper'),
            3 => __('3時間', 'wpap-card-helper'),
            6 => __('6時間', 'wpap-card-helper'),
            12 => __('12時間', 'wpap-card-helper'),
            24 => __('24時間（推奨）', 'wpap-card-helper'),
            48 => __('48時間', 'wpap-card-helper'),
            72 => __('72時間', 'wpap-card-helper'),
        );
        foreach ($choices as $value => $label) {
            $selected = ($cache_hours == $value) ? 'selected' : '';
            echo '<option value="' . esc_attr($value) . '" ' . $selected . '>' . esc_html($label) . '</option>';
        }
        echo '</select>';
        echo '<p class="description">' . esc_html__('商品情報（価格・在庫など）をキャッシュする時間です。短いとAPIリクエストが増えます。', 'wpap-card-helper') . '</p>';
    }
    
    /**
     * Credential IDフィールド
     */
    public function render_credential_id_field() {
        $options = get_option(self::OPTION_NAME);
        $value = isset($options['credential_id']) ? $options['credential_id'] : '';
        echo '<input type="text" name="' . esc_attr(self::OPTION_NAME) . '[credential_id]" value="' . esc_attr($value) . '" class="regular-text" autocomplete="off">';
        echo '<p class="description">Credential ID</p>';
    }

    /**
     * Credential Secretフィールド
     */
    public function render_credential_secret_field() {
        $options = get_option(self::OPTION_NAME);
        $value = isset($options['credential_secret']) ? $options['credential_secret'] : '';
        echo '<input type="password" name="' . esc_attr(self::OPTION_NAME) . '[credential_secret]" value="' . esc_attr($value) . '" class="regular-text" autocomplete="off">';
        echo '<p class="description">Secret</p>';
    }

    /**
     * トラッキングIDフィールド
     */
    public function render_tracking_id_field() {
        $options = get_option(self::OPTION_NAME);
        $value = isset($options['tracking_id']) ? $options['tracking_id'] : '';
        /* translators: %s: example tracking ID */
        $placeholder = sprintf(__('例: %s', 'wpap-card-helper'), 'yourtag-22');
        echo '<input type="text" name="' . esc_attr(self::OPTION_NAME) . '[tracking_id]" value="' . esc_attr($value) . '" class="regular-text" placeholder="' . esc_attr($placeholder) . '">';
        echo '<p class="description">' . esc_html__('AmazonアソシエイトのトラッキングID（例: yourtag-22）を入力してください。', 'wpap-card-helper') . '</p>';
    }
    
    /**
     * 設定のサニタイズ
     */
    public function sanitize_settings($input) {
        $sanitized = array();
        
        // API無効化オプション
        $sanitized['disable_api'] = isset($input['disable_api']) && $input['disable_api'] ? 1 : 0;
        
        if (isset($input['credential_id'])) {
            $sanitized['credential_id'] = sanitize_text_field($input['credential_id']);
        }
        
        if (isset($input['credential_secret'])) {
            $sanitized['credential_secret'] = sanitize_text_field($input['credential_secret']);
        }
        
        if (isset($input['tracking_id'])) {
            $sanitized['tracking_id'] = sanitize_text_field($input['tracking_id']);
        }
        
        // キャッシュ時間
        if (isset($input['cache_hours'])) {
            $sanitized['cache_hours'] = intval($input['cache_hours']);
        }
        
        // トークンキャッシュをクリア（クレデンシャルが変更された場合）
        delete_transient('wpap_replacer_token');
        
        // 商品キャッシュをすべてクリア
        global $wpdb;
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s OR option_name LIKE %s",
                '_transient_wpap_product_%',
                '_transient_timeout_wpap_product_%'
            )
        );
        
        return $sanitized;
    }
    
    /**
     * 設定ページの描画
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <form action="options.php" method="post">
                <?php
                settings_fields('wpap_replacer_options');
                do_settings_sections('wpap-card-helper');
                submit_button(__('設定を保存', 'wpap-card-helper'));
                ?>
            </form>
            
            <hr>

            <h2><?php esc_html_e('API接続テスト', 'wpap-card-helper'); ?></h2>
            <p><?php esc_html_e('設定を保存した後、以下のボタンでAPI接続をテストできます。', 'wpap-card-helper'); ?></p>
            <button type="button" id="wpap-test-api" class="button button-secondary"><?php esc_html_e('接続テスト', 'wpap-card-helper'); ?></button>
            <div id="wpap-test-result" style="margin-top: 10px;"></div>
            
            <script>
            jQuery(document).ready(function($) {
                $('#wpap-test-api').on('click', function() {
                    var $button = $(this);
                    var $result = $('#wpap-test-result');
                    
                    $button.prop('disabled', true).text('テスト中...');
                    $result.html('<p>接続をテストしています...</p>');
                    
                    $.ajax({
                        url: '<?php echo rest_url('wpap-card-helper/v1/test'); ?>',
                        method: 'POST',
                        beforeSend: function(xhr) {
                            xhr.setRequestHeader('X-WP-Nonce', '<?php echo wp_create_nonce('wp_rest'); ?>');
                        },
                        success: function(response) {
                            if (response.success) {
                                $result.html('<div class="notice notice-success"><p>✅ ' + response.message + '</p></div>');
                            } else {
                                $result.html('<div class="notice notice-error"><p>❌ ' + response.message + '</p></div>');
                            }
                        },
                        error: function(xhr) {
                            var message = xhr.responseJSON ? xhr.responseJSON.message : 'エラーが発生しました';
                            $result.html('<div class="notice notice-error"><p>❌ ' + message + '</p></div>');
                        },
                        complete: function() {
                            $button.prop('disabled', false).text('接続テスト');
                        }
                    });
                });
            });
            </script>
        </div>
        <?php
    }
    
    /**
     * 設定値の取得
     */
    public static function get_setting($key) {
        $options = get_option(self::OPTION_NAME);
        return isset($options[$key]) ? $options[$key] : '';
    }
    
    /**
     * すべての設定を取得
     */
    public static function get_all_settings() {
        return get_option(self::OPTION_NAME, array(
            'credential_id' => '',
            'credential_secret' => '',
            'tracking_id' => '',
        ));
    }
}
