<?php
/**
 * REST API エンドポイントクラス
 */

if (!defined('ABSPATH')) {
    exit;
}

class WPAP_Card_Helper_REST_API {

    /**
     * APIネームスペース
     */
    const NAMESPACE = 'wpap-card-helper/v1';

    /**
     * ショートコード属性用のエスケープ
     *
     * @param string $value 属性値
     * @return string エスケープされた値
     */
    private function escape_shortcode_attr($value) {
        if (empty($value)) {
            return $value;
        }
        // ショートコード属性として安全な形式にエスケープ
        $value = htmlspecialchars($value, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        $value = str_replace('[', '&#91;', $value);
        $value = str_replace(']', '&#93;', $value);
        return $value;
    }

    /**
     * コンストラクタ
     */
    public function __construct() {
        add_action('rest_api_init', array($this, 'register_routes'));
    }
    
    /**
     * REST APIルートの登録
     */
    public function register_routes() {
        // ショートコードスキャン
        register_rest_route(self::NAMESPACE, '/scan', array(
            'methods' => 'POST',
            'callback' => array($this, 'scan_shortcodes'),
            'permission_callback' => array($this, 'check_permission'),
        ));
        
        // ショートコード置換
        register_rest_route(self::NAMESPACE, '/replace', array(
            'methods' => 'POST',
            'callback' => array($this, 'replace_shortcodes'),
            'permission_callback' => array($this, 'check_permission'),
        ));
        
        // 元に戻す
        register_rest_route(self::NAMESPACE, '/revert', array(
            'methods' => 'POST',
            'callback' => array($this, 'revert_shortcodes'),
            'permission_callback' => array($this, 'check_permission'),
        ));
        
        // API接続テスト
        register_rest_route(self::NAMESPACE, '/test', array(
            'methods' => 'POST',
            'callback' => array($this, 'test_api'),
            'permission_callback' => array($this, 'check_permission'),
        ));
        
        // 単一商品情報取得
        register_rest_route(self::NAMESPACE, '/product/(?P<asin>[A-Z0-9]+)', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_product'),
            'permission_callback' => array($this, 'check_permission'),
        ));
        
        // スクレイピングによるタイトル取得（API不要）
        register_rest_route(self::NAMESPACE, '/scrape-title/(?P<asin>[A-Z0-9]+)', array(
            'methods' => 'GET',
            'callback' => array($this, 'scrape_product_title'),
            'permission_callback' => array($this, 'check_permission'),
        ));
    }
    
    /**
     * 権限チェック
     */
    public function check_permission() {
        return current_user_can('edit_posts');
    }
    
    /**
     * ショートコードのスキャン
     */
    public function scan_shortcodes($request) {
        $post_id = $request->get_param('post_id');
        
        if (empty($post_id)) {
            return new WP_Error('missing_post_id', __('投稿IDが指定されていません。', 'wpap-card-helper'), array('status' => 400));
        }
        
        $post = get_post($post_id);
        
        if (!$post) {
            return new WP_Error('invalid_post', __('指定された投稿が見つかりません。', 'wpap-card-helper'), array('status' => 404));
        }
        
        $processor = new WPAP_Card_Helper_Shortcode_Processor();
        $shortcodes = $processor->scan_shortcodes($post->post_content);
        
        // 未置換と置換済みを分類
        $unreplaced = array_filter($shortcodes, function($sc) {
            return !$sc['is_replaced'];
        });
        
        $replaced = array_filter($shortcodes, function($sc) {
            return $sc['is_replaced'];
        });
        
        return rest_ensure_response(array(
            'success' => true,
            'data' => array(
                'total' => count($shortcodes),
                'unreplaced_count' => count($unreplaced),
                'replaced_count' => count($replaced),
                'shortcodes' => array_values($shortcodes),
            ),
        ));
    }
    
    /**
     * ショートコードの置換
     */
    public function replace_shortcodes($request) {
        $post_id = $request->get_param('post_id');
        $asins = $request->get_param('asins'); // 特定のASINのみ置換する場合
        
        if (empty($post_id)) {
            return new WP_Error('missing_post_id', __('投稿IDが指定されていません。', 'wpap-card-helper'), array('status' => 400));
        }
        
        $post = get_post($post_id);
        
        if (!$post) {
            return new WP_Error('invalid_post', __('指定された投稿が見つかりません。', 'wpap-card-helper'), array('status' => 404));
        }
        
        // 設定取得
        $settings = WPAP_Card_Helper_Settings::get_all_settings();

        // トラッキングIDのチェック（ショートコードレンダリング時に必要）
        if (empty($settings['tracking_id'])) {
            return new WP_Error('missing_tracking_id', __('トラッキングIDが設定されていません。設定画面で設定してください。', 'wpap-card-helper'), array('status' => 400));
        }
        
        $processor = new WPAP_Card_Helper_Shortcode_Processor();
        $shortcodes = $processor->scan_shortcodes($post->post_content);
        
        // 未置換のショートコードのみ対象
        $unreplaced = array_filter($shortcodes, function($sc) use ($asins) {
            if ($sc['is_replaced']) {
                return false;
            }
            if (!empty($asins) && !in_array($sc['asin'], $asins)) {
                return false;
            }
            return true;
        });
        
        if (empty($unreplaced)) {
            return rest_ensure_response(array(
                'success' => true,
                'message' => __('置換対象のショートコードがありません。', 'wpap-card-helper'),
                'data' => array(
                    'replaced_count' => 0,
                ),
            ));
        }
        
        $content = $post->post_content;
        $replaced_count = 0;
        
        // 常にショートコード形式で置換
        foreach ($unreplaced as $shortcode_info) {
            $asin = esc_attr($shortcode_info['asin']);
            $title = $this->escape_shortcode_attr($shortcode_info['title'] ?: '');

            $new_shortcode = '[wpap_card asin="' . $asin . '"';
            if (!empty($title)) {
                $new_shortcode .= ' title="' . $title . '"';
            }
            $new_shortcode .= ']';

            $content = $processor->replace_shortcode($content, $shortcode_info['shortcode'], $new_shortcode);
            $replaced_count++;
        }
        
        // 投稿を更新
        $result = wp_update_post(array(
            'ID' => $post_id,
            'post_content' => $content,
        ), true);
        
        if (is_wp_error($result)) {
            return $result;
        }
        
        return rest_ensure_response(array(
            'success' => true,
            /* translators: %d: number of replaced shortcodes */
            'message' => sprintf(__('%d件のショートコードを置換しました。', 'wpap-card-helper'), $replaced_count),
            'data' => array(
                'replaced_count' => $replaced_count,
            ),
        ));
    }
    
    /**
     * 元に戻す
     */
    public function revert_shortcodes($request) {
        $post_id = $request->get_param('post_id');
        $asin = $request->get_param('asin'); // 特定のASINのみ復元する場合
        
        if (empty($post_id)) {
            return new WP_Error('missing_post_id', __('投稿IDが指定されていません。', 'wpap-card-helper'), array('status' => 400));
        }
        
        $post = get_post($post_id);
        
        if (!$post) {
            return new WP_Error('invalid_post', __('指定された投稿が見つかりません。', 'wpap-card-helper'), array('status' => 404));
        }
        
        $processor = new WPAP_Card_Helper_Shortcode_Processor();
        
        if (!empty($asin)) {
            $content = $processor->revert_by_asin($post->post_content, $asin);
        } else {
            $content = $processor->revert_all($post->post_content);
        }
        
        // 変更がない場合
        if ($content === $post->post_content) {
            return rest_ensure_response(array(
                'success' => true,
                'message' => __('復元対象のカードがありません。', 'wpap-card-helper'),
                'data' => array(
                    'reverted' => false,
                ),
            ));
        }
        
        // 投稿を更新
        $result = wp_update_post(array(
            'ID' => $post_id,
            'post_content' => $content,
        ), true);
        
        if (is_wp_error($result)) {
            return $result;
        }
        
        return rest_ensure_response(array(
            'success' => true,
            'message' => __('元のショートコードに復元しました。', 'wpap-card-helper'),
            'data' => array(
                'reverted' => true,
            ),
        ));
    }

    /**
     * API接続テスト
     */
    public function test_api($request) {
        $amazon_api = new WPAP_Card_Helper_Amazon_API();
        $result = $amazon_api->test_connection();
        
        return rest_ensure_response($result);
    }
    
    /**
     * 単一商品情報の取得
     */
    public function get_product($request) {
        $asin = $request->get_param('asin');

        if (empty($asin)) {
            return new WP_Error('missing_asin', __('ASINが指定されていません。', 'wpap-card-helper'), array('status' => 400));
        }

        // ASIN形式の検証（10桁の英数字）
        if (!preg_match('/^[A-Z0-9]{10}$/i', $asin)) {
            return new WP_Error('invalid_asin', __('ASINの形式が正しくありません。10桁の英数字で指定してください。', 'wpap-card-helper'), array('status' => 400));
        }

        $amazon_api = new WPAP_Card_Helper_Amazon_API();
        $products = $amazon_api->get_items($asin);

        if (is_wp_error($products)) {
            return $products;
        }

        if (!isset($products[$asin])) {
            return new WP_Error('product_not_found', __('商品が見つかりませんでした。', 'wpap-card-helper'), array('status' => 404));
        }
        
        return rest_ensure_response(array(
            'success' => true,
            'data' => $products[$asin],
        ));
    }
    
    /**
     * 商品情報取得（API優先、フォールバックでスクレイピング）
     */
    public function scrape_product_title($request) {
        $asin = $request->get_param('asin');

        if (empty($asin)) {
            return new WP_Error('missing_asin', __('ASINが指定されていません。', 'wpap-card-helper'), array('status' => 400));
        }

        // ASIN形式の検証（10桁の英数字）
        if (!preg_match('/^[A-Z0-9]{10}$/i', $asin)) {
            return new WP_Error('invalid_asin', __('ASINの形式が正しくありません。10桁の英数字で指定してください。', 'wpap-card-helper'), array('status' => 400));
        }

        // API設定をチェック
        $settings = WPAP_Card_Helper_Settings::get_all_settings();
        $disable_api = isset($settings['disable_api']) && $settings['disable_api'];
        $has_api_credentials = !empty($settings['credential_id']) && !empty($settings['credential_secret']);
        
        // APIが有効で資格情報がある場合はAPIを使用
        if (!$disable_api && $has_api_credentials) {
            $result = $this->fetch_from_api($asin);
            if (!is_wp_error($result)) {
                return rest_ensure_response($result);
            }
            // APIエラーの場合はスクレイピングにフォールバック
        }
        
        // スクレイピングで取得
        return $this->fetch_from_scraping($asin);
    }
    
    /**
     * APIから商品情報を取得
     */
    private function fetch_from_api($asin) {
        $amazon_api = new WPAP_Card_Helper_Amazon_API();
        $products = $amazon_api->get_items($asin);
        
        if (is_wp_error($products)) {
            return $products;
        }
        
        if (!isset($products[$asin])) {
            return new WP_Error('product_not_found', '商品が見つかりませんでした。', array('status' => 404));
        }
        
        $product = $products[$asin];
        
        // タイトルが空の場合はエラーとして返す（スクレイピングにフォールバックさせる）
        if (empty($product['title'])) {
            return new WP_Error('empty_title', __('APIからタイトルを取得できませんでした。', 'wpap-card-helper'), array('status' => 404));
        }
        
        return array(
            'success' => true,
            'source' => 'api',
            'data' => array(
                'asin' => $asin,
                'title' => $product['title'],
                'price' => $product['price'] ?? '',
                'image' => $product['image'] ?? '',
            ),
        );
    }
    
    /**
     * スクレイピングで商品情報を取得
     */
    private function fetch_from_scraping($asin) {
        // Amazon商品ページURL
        $url = 'https://www.amazon.co.jp/dp/' . $asin;
        
        // ページデータ取得
        $response = wp_remote_get($url, array(
            'timeout' => 15,
            'headers' => array(
                'User-Agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            ),
        ));
        
        if (is_wp_error($response)) {
            /* translators: %s: error message */
            return new WP_Error('fetch_error', sprintf(__('ページの取得に失敗しました: %s', 'wpap-card-helper'), $response->get_error_message()), array('status' => 500));
        }

        $body = wp_remote_retrieve_body($response);

        if (empty($body)) {
            return new WP_Error('empty_response', __('ページの内容が空です。', 'wpap-card-helper'), array('status' => 500));
        }

        // タイトルを抽出
        $title = $this->extract_title_from_html($body);

        // 価格を抽出
        $price = $this->extract_price_from_html($body);

        if (empty($title)) {
            return new WP_Error('title_not_found', __('タイトルが見つかりませんでした。ASINを確認してください。', 'wpap-card-helper'), array('status' => 404));
        }
        
        return rest_ensure_response(array(
            'success' => true,
            'source' => 'scraping',
            'data' => array(
                'asin' => $asin,
                'title' => $title,
                'price' => $price,
            ),
        ));
    }
    
    /**
     * HTMLからタイトルを抽出
     */
    private function extract_title_from_html($body) {
        $title = '';
        
        // パターン1: productTitleというIDの要素（最も正確）
        if (preg_match('/<span id="productTitle"[^>]*>\s*(.+?)\s*<\/span>/s', $body, $matches)) {
            $title = strip_tags($matches[1]);
        }
        
        // パターン2: Open Graphタグ
        if (empty($title) && preg_match('/<meta property="og:title" content="([^"]+)"/', $body, $matches)) {
            $title = $matches[1];
        }
        
        // パターン3: titleタグ
        if (empty($title) && preg_match('/<title>\s*(.+?)\s*<\/title>/s', $body, $matches)) {
            $title = $matches[1];
            $title = preg_replace('/^Amazon\.co\.jp[：:]\s*/', '', $title);
            $title = preg_replace('/\s*[：:]\s*.+$/', '', $title);
        }
        
        if (!empty($title)) {
            $title = html_entity_decode($title, ENT_QUOTES | ENT_HTML5, 'UTF-8');
            $title = trim($title);
            $title = preg_replace('/\s+/', ' ', $title);
        }
        
        return $title;
    }
    
    /**
     * HTMLから価格を抽出
     */
    private function extract_price_from_html($body) {
        $price = '';
        
        // パターン1: apex_desktop (新しいレイアウト)
        if (preg_match('/<span class="a-price-whole"[^>]*>([0-9,]+)<\/span>/', $body, $matches)) {
            $price = '¥' . $matches[1];
        }
        
        // パターン2: priceblock
        if (empty($price) && preg_match('/<span id="priceblock_ourprice"[^>]*>([^<]+)<\/span>/', $body, $matches)) {
            $price = trim($matches[1]);
        }
        
        // パターン3: corePrice
        if (empty($price) && preg_match('/<span class="a-offscreen">([^<]+)<\/span>/', $body, $matches)) {
            $price = trim($matches[1]);
            // 最初に見つかった価格のみ使用（通常は販売価格）
            if (preg_match('/[¥￥][0-9,]+/', $price, $priceMatch)) {
                $price = $priceMatch[0];
            }
        }
        
        if (!empty($price)) {
            $price = html_entity_decode($price, ENT_QUOTES | ENT_HTML5, 'UTF-8');
            $price = trim($price);
        }
        
        return $price;
    }
}

