<?php
/**
 * 新しいショートコードクラス
 * [wpap_card] ショートコードを提供
 */

if (!defined('ABSPATH')) {
    exit;
}

class WPAP_Card_Helper_New_Shortcode {
    
    /**
     * コンストラクタ
     */
    public function __construct() {
        add_shortcode('wpap_card', array($this, 'render_shortcode'));
    }
    
    /**
     * ショートコード属性のHTMLエンティティをデコード
     *
     * @param string $value 属性値
     * @return string デコードされた値
     */
    private function decode_shortcode_attr($value) {
        if (empty($value)) {
            return $value;
        }
        // HTMLエンティティをデコード（&#91; → [, &lt; → < など）
        return html_entity_decode($value, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    }

    /**
     * ショートコードをレンダリング
     *
     * 使用例:
     * [wpap_card asin="B08N68GBQD" title="商品名"]
     * [wpap_card asin="B08N68GBQD"]
     *
     * @param array $atts ショートコード属性
     * @return string HTML出力
     */
    public function render_shortcode($atts) {
        $atts = shortcode_atts(array(
            'asin' => '',
            'title' => '',
            'price' => '',
            'id' => '', // asinの別名としても受け付ける
        ), $atts, 'wpap_card');

        // 属性値のHTMLエンティティをデコード
        $atts['title'] = $this->decode_shortcode_attr($atts['title']);
        $atts['price'] = $this->decode_shortcode_attr($atts['price']);

        // ASINを取得（asinまたはid属性から）
        $asin = !empty($atts['asin']) ? $atts['asin'] : $atts['id'];
        
        if (empty($asin)) {
            return '<!-- wpap_card: ASINが指定されていません -->';
        }
        
        // 設定を取得
        $settings = WPAP_Card_Helper_Settings::get_all_settings();
        $tracking_id = $settings['tracking_id'] ?? '';
        $disable_api = isset($settings['disable_api']) && $settings['disable_api'];
        
        if (empty($tracking_id)) {
            return '<!-- wpap_card: トラッキングIDが設定されていません -->';
        }
        
        $card_generator = new WPAP_Card_Helper_Card_Generator();
        
        if ($disable_api) {
            // API無効化モード: シンプルカードを生成
            $product_data = array(
                'asin' => $asin,
                'title' => !empty($atts['title']) ? $atts['title'] : 'Amazon商品',
                'url' => 'https://www.amazon.co.jp/dp/' . $asin . '?tag=' . $tracking_id,
                'image' => '',
                'price' => !empty($atts['price']) ? $atts['price'] : '',
                'availability' => '',
            );
            
            return $card_generator->generate_simple_card($product_data);
        } else {
            // API使用モード: 商品情報を取得
            $cache_key = 'wpap_product_' . $asin;
            $cached_data = get_transient($cache_key);
            
            if ($cached_data !== false) {
                return $card_generator->generate_card($cached_data);
            }
            
            $amazon_api = new WPAP_Card_Helper_Amazon_API();
            $products = $amazon_api->get_items($asin);
            
            if (is_wp_error($products) || !isset($products[$asin])) {
                // API取得失敗時はシンプルカードにフォールバック
                $product_data = array(
                    'asin' => $asin,
                    'title' => !empty($atts['title']) ? $atts['title'] : 'Amazon商品',
                    'url' => 'https://www.amazon.co.jp/dp/' . $asin . '?tag=' . $tracking_id,
                    'image' => '',
                    'price' => !empty($atts['price']) ? $atts['price'] : '',
                    'availability' => '',
                );
                
                return $card_generator->generate_simple_card($product_data);
            }
            
            $product_data = $products[$asin];
            
            // キャッシュ時間を設定から取得（デフォルト24時間）
            $cache_hours = isset($settings['cache_hours']) ? intval($settings['cache_hours']) : 24;
            $cache_seconds = $cache_hours * HOUR_IN_SECONDS;
            set_transient($cache_key, $product_data, $cache_seconds);
            
            return $card_generator->generate_card($product_data);
        }
    }
}
