<?php
/**
 * エディタ統合クラス
 * ClassicエディタとGutenbergエディタに商品カード挿入機能を追加
 */

if (!defined('ABSPATH')) {
    exit;
}

class WPAP_Card_Helper_Editor_Integration {
    
    /**
     * コンストラクタ
     */
    public function __construct() {
        // Classicエディタ統合
        add_action('media_buttons', array($this, 'add_classic_editor_button'));
        add_action('admin_footer', array($this, 'add_classic_editor_dialog'));
        
        // Gutenbergブロック登録
        add_action('init', array($this, 'register_gutenberg_block'));
        
        // エディタ用スクリプト・スタイル
        add_action('admin_enqueue_scripts', array($this, 'enqueue_editor_assets'));
    }
    
    /**
     * Classicエディタボタン追加
     */
    public function add_classic_editor_button($editor_id) {
        // 投稿編集画面のみ
        $screen = get_current_screen();
        if (!$screen || !in_array($screen->base, array('post', 'page'))) {
            return;
        }
        
        echo '<button type="button" id="wpap-insert-card-btn" class="button" style="margin-left: 5px;">';
        echo '<span class="dashicons dashicons-cart" style="vertical-align: middle;"></span> ';
        echo 'Amazon商品カード';
        echo '</button>';
    }
    
    /**
     * Classicエディタダイアログ追加
     */
    public function add_classic_editor_dialog() {
        $screen = get_current_screen();
        if (!$screen || !in_array($screen->base, array('post', 'page'))) {
            return;
        }
        ?>
        <div id="wpap-insert-card-dialog" style="display: none;">
            <div style="background: #fff; padding: 20px; max-width: 500px;">
                <h2 style="margin-top: 0;">Amazon商品カード挿入</h2>
                
                <p>
                    <label for="wpap-card-asin" style="display: block; margin-bottom: 5px;">
                        <strong>ASIN:</strong>
                    </label>
                    <div style="display: flex; gap: 5px;">
                        <input type="text" id="wpap-card-asin" class="widefat" placeholder="例: B08N68GBQD" style="flex: 1;" />
                        <button type="button" id="wpap-fetch-title-btn" class="button">
                            <span class="dashicons dashicons-update" style="vertical-align: middle;"></span> 情報取得
                        </button>
                    </div>
                    <span class="description">Amazonの商品識別コード（10桁）を入力してください。</span>
                    <span id="wpap-fetch-status" style="display: none; margin-top: 5px; font-size: 12px;"></span>
                </p>
                
                <p>
                    <label for="wpap-card-title" style="display: block; margin-bottom: 5px;">
                        <strong>商品名:</strong>
                    </label>
                    <input type="text" id="wpap-card-title" class="widefat" placeholder="例: ワイヤレスイヤホン" />
                </p>
                
                <p>
                    <label for="wpap-card-price" style="display: block; margin-bottom: 5px;">
                        <strong>価格:</strong>
                    </label>
                    <input type="text" id="wpap-card-price" class="widefat" placeholder="例: ¥3,980" />
                    <span class="description">価格はショートコードに含まれます。</span>
                </p>
                
                <p style="margin-top: 20px;">
                    <button type="button" id="wpap-insert-card-submit" class="button button-primary">
                        挿入
                    </button>
                    <button type="button" id="wpap-insert-card-cancel" class="button" style="margin-left: 10px;">
                        キャンセル
                    </button>
                </p>
            </div>
        </div>
        
        <style>
        #wpap-insert-card-dialog {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.7);
            z-index: 100000;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        #wpap-insert-card-dialog > div {
            box-shadow: 0 5px 15px rgba(0,0,0,0.3);
            border-radius: 4px;
        }
        .dashicons.spin {
            animation: wpap-spin 1s linear infinite;
        }
        @keyframes wpap-spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
        </style>
        
        <script>
        jQuery(document).ready(function($) {
            // ダイアログを開く
            $('#wpap-insert-card-btn').on('click', function(e) {
                e.preventDefault();
                $('#wpap-insert-card-dialog').show();
                $('#wpap-card-asin').focus();
            });
            
            // キャンセル
            $('#wpap-insert-card-cancel').on('click', function() {
                $('#wpap-insert-card-dialog').hide();
                $('#wpap-card-asin').val('');
                $('#wpap-card-title').val('');
                $('#wpap-card-price').val('');
            });
            
            // Escキーでキャンセル
            $(document).on('keydown', function(e) {
                if (e.key === 'Escape' && $('#wpap-insert-card-dialog').is(':visible')) {
                    $('#wpap-insert-card-cancel').click();
                }
            });
            
            // タイトル取得
            $('#wpap-fetch-title-btn').on('click', function() {
                var $btn = $(this);
                var $status = $('#wpap-fetch-status');
                var asin = $('#wpap-card-asin').val().trim();
                
                if (!asin) {
                    alert('ASINを入力してください。');
                    return;
                }
                
                // ボタンを無効化
                $btn.prop('disabled', true).find('.dashicons').addClass('spin');
                $status.show().css('color', '#666').text('取得中...');
                
                // REST APIで商品情報取得（スクレイピング）
                $.ajax({
                    url: '<?php echo rest_url('wpap-card-helper/v1/scrape-title/'); ?>' + asin,
                    method: 'GET',
                    beforeSend: function(xhr) {
                        xhr.setRequestHeader('X-WP-Nonce', '<?php echo wp_create_nonce('wp_rest'); ?>');
                    },
                    success: function(response) {
                        if (response.success && response.data) {
                            if (response.data.title) {
                                $('#wpap-card-title').val(response.data.title);
                            }
                            if (response.data.price) {
                                $('#wpap-card-price').val(response.data.price);
                            }
                            var source = response.source === 'api' ? 'API' : 'スクレイピング';
                            $status.css('color', 'green').text('✓ ' + source + 'で情報を取得しました');
                        } else {
                            $status.css('color', 'orange').text('情報が取得できませんでした');
                        }
                    },
                    error: function(xhr) {
                        var message = xhr.responseJSON && xhr.responseJSON.message ? xhr.responseJSON.message : '情報の取得に失敗しました';
                        $status.css('color', 'red').text('✗ ' + message);
                    },
                    complete: function() {
                        $btn.prop('disabled', false).find('.dashicons').removeClass('spin');
                        setTimeout(function() {
                            $status.fadeOut();
                        }, 3000);
                    }
                });
            });
            
            // ショートコード属性用のエスケープ関数
            function escapeShortcodeAttr(str) {
                if (!str) return str;
                return str
                    .replace(/&/g, '&amp;')
                    .replace(/"/g, '&quot;')
                    .replace(/</g, '&lt;')
                    .replace(/>/g, '&gt;')
                    .replace(/\[/g, '&#91;')
                    .replace(/\]/g, '&#93;');
            }

            // 挿入
            $('#wpap-insert-card-submit').on('click', function() {
                var asin = $('#wpap-card-asin').val().trim();
                var title = $('#wpap-card-title').val().trim();

                if (!asin) {
                    alert('ASINを入力してください。');
                    return;
                }

                // ASINの簡易検証（10桁の英数字）
                if (!/^[A-Z0-9]{10}$/i.test(asin)) {
                    if (!confirm('ASINは通常10桁の英数字です。このまま挿入しますか？')) {
                        return;
                    }
                }

                // ショートコード生成（特殊文字をエスケープ）
                var shortcode = '[wpap_card asin="' + asin + '"';
                if (title) {
                    shortcode += ' title="' + escapeShortcodeAttr(title) + '"';
                }
                var price = $('#wpap-card-price').val().trim();
                if (price) {
                    shortcode += ' price="' + escapeShortcodeAttr(price) + '"';
                }
                shortcode += ']';
                
                // エディタに挿入
                if (typeof tinymce !== 'undefined' && tinymce.activeEditor && !tinymce.activeEditor.isHidden()) {
                    // ビジュアルエディタ
                    tinymce.activeEditor.execCommand('mceInsertContent', false, shortcode);
                } else {
                    // テキストエディタ
                    var $textarea = $('#content');
                    if ($textarea.length) {
                        var cursorPos = $textarea[0].selectionStart;
                        var textBefore = $textarea.val().substring(0, cursorPos);
                        var textAfter = $textarea.val().substring(cursorPos);
                        $textarea.val(textBefore + shortcode + textAfter);
                        $textarea[0].selectionStart = $textarea[0].selectionEnd = cursorPos + shortcode.length;
                        $textarea.focus();
                    }
                }
                
                // ダイアログを閉じる
                $('#wpap-insert-card-cancel').click();
            });
            
            // Enterキーで挿入
            $('#wpap-card-asin, #wpap-card-title').on('keypress', function(e) {
                if (e.which === 13) {
                    e.preventDefault();
                    $('#wpap-insert-card-submit').click();
                }
            });
        });
        </script>
        <?php
    }
    
    /**
     * エディタアセットの読み込み
     */
    public function enqueue_editor_assets($hook) {
        if (!in_array($hook, array('post.php', 'post-new.php'))) {
            return;
        }
        
        // 必要に応じて追加のスタイル・スクリプトを読み込み
    }
    
    /**
     * Gutenbergブロック登録
     */
    public function register_gutenberg_block() {
        // Gutenbergブロックの登録
        register_block_type('wpap-card-helper/product-card', array(
            'editor_script' => 'wpap-card-helper-block-editor',
            'editor_style' => 'wpap-card-helper-block-editor-style',
            'render_callback' => array($this, 'render_block'),
        ));
        
        // ブロックエディタ用スクリプト登録
        wp_register_script(
            'wpap-card-helper-block-editor',
            WPAP_CARD_HELPER_PLUGIN_URL . 'assets/js/gutenberg-block.js',
            array('wp-blocks', 'wp-element', 'wp-editor', 'wp-components', 'wp-i18n'),
            WPAP_CARD_HELPER_VERSION,
            true
        );
    }
    
    /**
     * ブロックのレンダリング
     */
    public function render_block($attributes) {
        $asin = isset($attributes['asin']) ? sanitize_text_field($attributes['asin']) : '';
        $title = isset($attributes['title']) ? sanitize_text_field($attributes['title']) : '';
        
        if (empty($asin)) {
            return '<!-- Amazon商品カード: ASINが指定されていません -->';
        }
        
        // ショートコードとして処理
        $shortcode = '[wpap_card asin="' . esc_attr($asin) . '"';
        if (!empty($title)) {
            $shortcode .= ' title="' . esc_attr($title) . '"';
        }
        $shortcode .= '"]';
        
        return do_shortcode($shortcode);
    }
}
