<?php
/**
 * アフィリエイトカード生成クラス
 */

if (!defined('ABSPATH')) {
    exit;
}

class WPAP_Card_Helper_Card_Generator {
    
    /**
     * コンストラクタ
     */
    public function __construct() {
        // 必要に応じて初期化処理
    }
    
    /**
     * 商品カードHTMLの生成
     * 
     * @param array $product_data 商品データ
     * @return string カードHTML
     */
    public function generate_card($product_data) {
        $asin = esc_attr($product_data['asin'] ?? '');
        $title = esc_html($product_data['title'] ?? '商品名を取得できませんでした');
        $url = esc_url($product_data['url'] ?? '#');
        $image = esc_url($product_data['image'] ?? '');
        $price = esc_html($product_data['price'] ?? '');
        $availability = esc_html($product_data['availability'] ?? '');
        
        // 画像がない場合のプレースホルダー
        $image_html = '';
        if (!empty($image)) {
            $image_html = '<img src="' . $image . '" alt="' . $title . '" loading="lazy">';
        } else {
            $image_html = '<div class="wpap-card-no-image">No Image</div>';
        }
        
        // 価格表示
        $price_html = '';
        if (!empty($price)) {
            $price_html = '<p class="wpap-card-price">' . $price . '</p>';
        }
        
        // 在庫状況表示
        $availability_html = '';
        if (!empty($availability)) {
            $availability_html = '<p class="wpap-card-availability">' . $availability . '</p>';
        }
        
        $html = <<<HTML
<div class="wpap-replacer-card" data-asin="{$asin}">
    <a href="{$url}" target="_blank" rel="nofollow noopener sponsored">
        <div class="wpap-card-image">
            {$image_html}
        </div>
        <div class="wpap-card-content">
            <h4 class="wpap-card-title">{$title}</h4>
            {$price_html}
            {$availability_html}
            <span class="wpap-card-button">Amazonで見る</span>
        </div>
    </a>
</div>
HTML;
        
        return $html;
    }
    
    /**
     * エラー時のカードHTMLの生成
     * 
     * @param string $asin ASIN
     * @param string $error_message エラーメッセージ
     * @param string $original_title 元のタイトル（ショートコードから）
     * @return string エラーカードHTML
     */
    public function generate_error_card($asin, $error_message, $original_title = '') {
        $asin = esc_attr($asin);
        $error_message = esc_html($error_message);
        $tracking_id = esc_attr(WPAP_Card_Helper_Settings::get_setting('tracking_id'));
        
        $url = 'https://www.amazon.co.jp/dp/' . $asin;
        if (!empty($tracking_id)) {
            $url .= '?tag=' . $tracking_id;
        }
        $url = esc_url($url);
        
        $title = !empty($original_title) ? esc_html($original_title) : 'ASIN: ' . $asin;
        
        $html = <<<HTML
<div class="wpap-replacer-card wpap-card-error" data-asin="{$asin}">
    <a href="{$url}" target="_blank" rel="nofollow noopener sponsored">
        <div class="wpap-card-image">
            <div class="wpap-card-no-image">⚠️</div>
        </div>
        <div class="wpap-card-content">
            <h4 class="wpap-card-title">{$title}</h4>
            <p class="wpap-card-error-message">{$error_message}</p>
            <span class="wpap-card-button">Amazonで見る</span>
        </div>
    </a>
</div>
HTML;
        
        return $html;
    }
    
    /**
     * 複数の商品カードを一括生成
     * 
     * @param array $products_data 商品データの配列
     * @return array ASIN => カードHTML の連想配列
     */
    public function generate_cards($products_data) {
        $cards = array();
        
        foreach ($products_data as $asin => $product_data) {
            $cards[$asin] = $this->generate_card($product_data);
        }
        
        return $cards;
    }
    
    /**
     * シンプルカードHTMLの生成（API無効化時用）
     * 画像・価格なしのシンプルなテキストリンクカード
     * 
     * @param array $product_data 商品データ
     * @return string カードHTML
     */
    public function generate_simple_card($product_data) {
        $asin = esc_attr($product_data['asin'] ?? '');
        $title = esc_html($product_data['title'] ?? 'Amazon商品');
        $url = esc_url($product_data['url'] ?? '#');
        $price = esc_html($product_data['price'] ?? '');
        
        // 価格表示
        $price_html = '';
        if (!empty($price)) {
            $price_html = '<p class="wpap-card-price">' . $price . '</p>';
        }
        
        // 説明文（価格がある場合は表示しない）
        $note_html = empty($price) ? '<p class="wpap-card-simple-note">Amazonで詳細を確認</p>' : '';
        
        $html = <<<HTML
<div class="wpap-replacer-card wpap-card-simple" data-asin="{$asin}">
    <a href="{$url}" target="_blank" rel="nofollow noopener sponsored">
        <div class="wpap-card-image">
            <div class="wpap-card-amazon-icon">
                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512" width="48" height="48" fill="#ff9900">
                    <path d="M257.2 162.7c-48.7 1.8-169.5 15.5-169.5 117.5 0 109.5 138.3 114 183.5 43.2 6.5 10.2 35.4 37.5 45.3 46.8l56.8-56S341 288.9 341 261.4V114.3C341 89 316.5 32 228.7 32 140.7 32 94 87 94 136.3l73.5 6.8c16.3-49.5 54.2-49.5 54.2-49.5 40.7-.1 35.5 29.8 35.5 69.1zm0 86.8c0 80-84.2 68-84.2 17.2 0-47.2 50.5-56.7 84.2-57.8v40.6zm136 163.5c-7.7 10-70 67-174.5 67S34.2 408.5 9.7 379c-6.8-7.7 1-11.3 5.5-8.3C88.5 415.2 203 488.5 387.7 401c7.5-3.7 13.3 2 5.5 12zm39.8 2.2c-6.5 15.8-16 26.8-21.2 31-5.5 4.5-9.5 2.7-6.5-3.8s19.3-46.5 12.7-55c-6.5-8.3-37-4.3-48-3.2-10.8 1-13 2-14-.3-2.3-5.7 21.7-15.5 37.5-17.5 15.7-1.8 41-.8 46 5.7 3.7 5.1 0 27.1-6.5 43.1z"/>
                </svg>
            </div>
        </div>
        <div class="wpap-card-content">
            <h4 class="wpap-card-title">{$title}</h4>
            {$price_html}
            {$note_html}
            <span class="wpap-card-button">Amazonで見る →</span>
        </div>
    </a>
</div>
HTML;
        
        return $html;
    }
}
