<?php
/**
 * 一括処理ページクラス
 */

if (!defined('ABSPATH')) {
    exit;
}

class WPAP_Card_Helper_Bulk_Processor {

    /**
     * ショートコード属性用のエスケープ
     *
     * @param string $value 属性値
     * @return string エスケープされた値
     */
    private function escape_shortcode_attr($value) {
        if (empty($value)) {
            return $value;
        }
        // ショートコード属性として安全な形式にエスケープ
        $value = htmlspecialchars($value, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        $value = str_replace('[', '&#91;', $value);
        $value = str_replace(']', '&#93;', $value);
        return $value;
    }

    /**
     * コンストラクタ
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_bulk_menu'));
        add_action('wp_ajax_wpap_bulk_scan', array($this, 'ajax_bulk_scan'));
        add_action('wp_ajax_wpap_bulk_replace', array($this, 'ajax_bulk_replace'));
        add_action('wp_ajax_wpap_bulk_revert', array($this, 'ajax_bulk_revert'));
        add_action('wp_ajax_wpap_force_repair', array($this, 'ajax_force_repair'));
    }
    
    /**
     * 管理メニューに追加
     */
    public function add_bulk_menu() {
        add_management_page(
            __('WPAP 一括処理', 'wpap-card-helper'),
            __('WPAP 一括処理', 'wpap-card-helper'),
            'edit_posts',
            'wpap-bulk-processor',
            array($this, 'render_bulk_page')
        );
    }
    
    /**
     * 一括処理ページの描画
     */
    public function render_bulk_page() {
        ?>
        <div class="wrap">
            <h1>WPAP 一括処理</h1>
            <p>WPアソシエイトポストR2のショートコードを全記事から検出し、一括で置換します。</p>
            
            <div id="wpap-bulk-panel" style="background: #fff; padding: 20px; border: 1px solid #ccc; border-radius: 8px; max-width: 800px;">
                
                <h2>📊 全記事スキャン</h2>
                <p>まずは全記事をスキャンして、置換対象のショートコードを検出します。</p>
                <button type="button" id="wpap-bulk-scan-btn" class="button button-primary button-large">
                    🔍 全記事をスキャン
                </button>
                
                <div id="wpap-scan-results" style="margin-top: 20px; display: none;">
                    <h3>スキャン結果</h3>
                    <table class="widefat" style="max-width: 600px;">
                        <tbody>
                            <tr>
                                <td>対象記事数</td>
                                <td id="result-post-count">-</td>
                            </tr>
                            <tr>
                                <td>未置換ショートコード数</td>
                                <td id="result-unreplaced">-</td>
                            </tr>
                            <tr>
                                <td>置換済み数</td>
                                <td id="result-replaced">-</td>
                            </tr>
                        </tbody>
                    </table>
                    
                    <div id="wpap-post-list" style="margin-top: 15px; max-height: 300px; overflow-y: auto;">
                        <!-- 投稿リストがここに表示される -->
                    </div>
                </div>
                
                <hr style="margin: 30px 0;">
                
                <h2>✨ 一括置換</h2>
                <p>スキャンで検出されたショートコードを全て置換します。</p>
                <button type="button" id="wpap-bulk-replace-btn" class="button button-large" disabled>
                    ✨ 一括置換を実行
                </button>
                
                <div id="wpap-replace-progress" style="margin-top: 20px; display: none;">
                    <div style="background: #ddd; border-radius: 4px; overflow: hidden; height: 24px;">
                        <div id="wpap-progress-bar" style="width: 0%; height: 100%; background: linear-gradient(90deg, #0073aa, #00a0d2); transition: width 0.3s;"></div>
                    </div>
                    <p id="wpap-progress-text" style="text-align: center; margin-top: 10px;">処理中...</p>
                </div>
                
                <div id="wpap-replace-results" style="margin-top: 20px; display: none;">
                    <div style="background: #d4edda; border: 1px solid #c3e6cb; padding: 15px; border-radius: 8px;">
                        <strong>✅ 完了</strong>
                        <p id="wpap-replace-summary"></p>
                    </div>
                </div>
                
                <hr style="margin: 30px 0;">
                
                <h2>↩️ 一括で元に戻す</h2>
                <p>置換済みのショートコードを全て元の形式に戻します。</p>
                <button type="button" id="wpap-bulk-revert-btn" class="button button-large" disabled>
                    ↩️ 一括で元に戻す
                </button>
                
                <div id="wpap-revert-progress" style="margin-top: 20px; display: none;">
                    <div style="background: #ddd; border-radius: 4px; overflow: hidden; height: 24px;">
                        <div id="wpap-revert-bar" style="width: 0%; height: 100%; background: linear-gradient(90deg, #dc3545, #f39c12); transition: width 0.3s;"></div>
                    </div>
                    <p id="wpap-revert-text" style="text-align: center; margin-top: 10px;">処理中...</p>
                </div>
                
                <div id="wpap-revert-results" style="margin-top: 20px; display: none;">
                    <div style="background: #fff3cd; border: 1px solid #ffc107; padding: 15px; border-radius: 8px;">
                        <strong>↩️ 復元完了</strong>
                        <p id="wpap-revert-summary"></p>
                    </div>
                </div>
                
                <hr style="margin: 30px 0;">
                
                <h2>🔧 全記事強制修復・復元</h2>
                <p style="color: #d63384;">スキャン結果に関係なく、全ての記事を強制的に修復・復元します。<br>破損したコンテンツや残骸を一括で除去できます。</p>
                <button type="button" id="wpap-force-repair-btn" class="button button-large" style="background: #dc3545; border-color: #dc3545; color: #fff;">
                    🔧 全記事強制修復・復元
                </button>
                
                <div id="wpap-repair-progress" style="margin-top: 20px; display: none;">
                    <div style="background: #ddd; border-radius: 4px; overflow: hidden; height: 24px;">
                        <div id="wpap-repair-bar" style="width: 0%; height: 100%; background: linear-gradient(90deg, #dc3545, #fd7e14); transition: width 0.3s;"></div>
                    </div>
                    <p id="wpap-repair-text" style="text-align: center; margin-top: 10px;">処理中...</p>
                </div>
                
                <div id="wpap-repair-results" style="margin-top: 20px; display: none;">
                    <div style="background: #d4edda; border: 1px solid #c3e6cb; padding: 15px; border-radius: 8px;">
                        <strong>🔧 修復完了</strong>
                        <p id="wpap-repair-summary"></p>
                    </div>
                </div>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            var postsToProcess = [];
            var postsToRevert = [];
            
            // スキャンボタン
            $('#wpap-bulk-scan-btn').on('click', function() {
                var $btn = $(this);
                $btn.prop('disabled', true).text('スキャン中...');
                
                // UIを初期状態にリセット
                $('#wpap-scan-results').hide();
                $('#wpap-replace-progress').hide();
                $('#wpap-replace-results').hide();
                $('#wpap-revert-progress').hide();
                $('#wpap-revert-results').hide();
                $('#wpap-repair-progress').hide();
                $('#wpap-repair-results').hide();
                
                // プログレスバーをリセット
                $('#wpap-progress-bar').css('width', '0%');
                $('#wpap-revert-bar').css('width', '0%');
                $('#wpap-repair-bar').css('width', '0%');
                $('#wpap-progress-text').text('処理中...');
                $('#wpap-revert-text').text('処理中...');
                $('#wpap-repair-text').text('処理中...');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'wpap_bulk_scan',
                        nonce: '<?php echo wp_create_nonce('wpap_bulk_scan'); ?>'
                    },
                    success: function(response) {
                        $btn.prop('disabled', false).text('🔍 全記事をスキャン');
                        
                        if (response.success) {
                            var data = response.data;
                            $('#result-post-count').text(data.post_count + '件');
                            $('#result-unreplaced').text(data.unreplaced_count + '件');
                            $('#result-replaced').text(data.replaced_count + '件');
                            
                            postsToProcess = data.posts_with_unreplaced;
                            postsToRevert = data.posts_with_replaced;
                            
                            // 投稿リスト表示
                            var listHtml = '<ul style="margin: 0; padding-left: 20px;">';
                            data.posts.forEach(function(post) {
                                var status = post.unreplaced > 0 ? '⚠️' : '✅';
                                var escapedTitle = $('<div>').text(post.title).html();
                                var escapedLink = $('<div>').text(post.edit_link).html();
                                listHtml += '<li>' + status + ' <a href="' + escapedLink + '" target="_blank">' + escapedTitle + '</a> (未置換: ' + parseInt(post.unreplaced, 10) + ', 置換済み: ' + parseInt(post.replaced, 10) + ')</li>';
                            });
                            listHtml += '</ul>';
                            $('#wpap-post-list').html(listHtml);
                            
                            $('#wpap-scan-results').show();
                            
                            // ボタンのテキストを元に戻す
                            $('#wpap-bulk-replace-btn').text('✨ 一括置換を実行');
                            $('#wpap-bulk-revert-btn').text('↩️ 一括で元に戻す');
                            $('#wpap-force-repair-btn').text('🔧 全記事強制修復・復元').prop('disabled', false);
                            
                            // 置換ボタン有効化
                            if (data.unreplaced_count > 0) {
                                $('#wpap-bulk-replace-btn').prop('disabled', false);
                            } else {
                                $('#wpap-bulk-replace-btn').prop('disabled', true);
                            }
                            
                            // 元に戻すボタン有効化
                            if (data.replaced_count > 0) {
                                $('#wpap-bulk-revert-btn').prop('disabled', false);
                            } else {
                                $('#wpap-bulk-revert-btn').prop('disabled', true);
                            }
                        } else {
                            alert('エラー: ' + response.data);
                        }
                    },
                    error: function() {
                        $btn.prop('disabled', false).text('🔍 全記事をスキャン');
                        alert('通信エラーが発生しました。');
                    }
                });
            });
            
            // 一括置換ボタン
            $('#wpap-bulk-replace-btn').on('click', function() {
                if (postsToProcess.length === 0) {
                    alert('置換対象の記事がありません。');
                    return;
                }
                
                if (!confirm('全ての未置換ショートコードを置換します。よろしいですか？')) {
                    return;
                }
                
                var $btn = $(this);
                $btn.prop('disabled', true);
                $('#wpap-replace-progress').show();
                $('#wpap-replace-results').hide();
                
                var total = postsToProcess.length;
                var processed = 0;
                var replaced = 0;
                
                function processNext() {
                    if (postsToProcess.length === 0) {
                        // 完了
                        $('#wpap-progress-bar').css('width', '100%');
                        $('#wpap-progress-text').text('完了！');
                        $('#wpap-replace-summary').text(processed + '件の記事を処理し、' + replaced + '件のショートコードを置換しました。');
                        $('#wpap-replace-results').show();
                        $('#wpap-bulk-replace-btn').text('✨ 完了');
                        return;
                    }
                    
                    var postId = postsToProcess.shift();
                    var progress = Math.round((processed / total) * 100);
                    $('#wpap-progress-bar').css('width', progress + '%');
                    $('#wpap-progress-text').text('処理中... (' + processed + '/' + total + ')');
                    
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'wpap_bulk_replace',
                            nonce: '<?php echo wp_create_nonce('wpap_bulk_replace'); ?>',
                            post_id: postId
                        },
                        success: function(response) {
                            processed++;
                            if (response.success) {
                                replaced += response.data.replaced_count || 0;
                            }
                            processNext();
                        },
                        error: function() {
                            processed++;
                            processNext();
                        }
                    });
                }
                
                processNext();
            });
            
            // 一括で元に戻すボタン
            $('#wpap-bulk-revert-btn').on('click', function() {
                if (postsToRevert.length === 0) {
                    alert('元に戻す対象の記事がありません。');
                    return;
                }
                
                if (!confirm('全ての置換済みショートコードを元に戻します。よろしいですか？')) {
                    return;
                }
                
                var $btn = $(this);
                $btn.prop('disabled', true);
                $('#wpap-revert-progress').show();
                $('#wpap-revert-results').hide();
                
                var total = postsToRevert.length;
                var processed = 0;
                var reverted = 0;
                var postsToRevertCopy = postsToRevert.slice();
                
                function revertNext() {
                    if (postsToRevertCopy.length === 0) {
                        // 完了
                        $('#wpap-revert-bar').css('width', '100%');
                        $('#wpap-revert-text').text('完了！');
                        $('#wpap-revert-summary').text(processed + '件の記事を処理し、' + reverted + '件のショートコードを元に戻しました。');
                        $('#wpap-revert-results').show();
                        $btn.text('↩️ 完了');
                        return;
                    }
                    
                    var postId = postsToRevertCopy.shift();
                    var progress = Math.round((processed / total) * 100);
                    $('#wpap-revert-bar').css('width', progress + '%');
                    $('#wpap-revert-text').text('処理中... (' + processed + '/' + total + ')');
                    
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'wpap_bulk_revert',
                            nonce: '<?php echo wp_create_nonce('wpap_bulk_revert'); ?>',
                            post_id: postId
                        },
                        success: function(response) {
                            processed++;
                            if (response.success) {
                                reverted += response.data.reverted_count || 0;
                            }
                            revertNext();
                        },
                        error: function() {
                            processed++;
                            revertNext();
                        }
                    });
                }
                
                revertNext();
            });
            
            // 強制修復ボタン
            $('#wpap-force-repair-btn').on('click', function() {
                if (!confirm('全ての記事を強制的に修復・復元します。\nこの操作は破損したコンテンツを修復し、全てのショートコードを元の形式に戻します。\n\nよろしいですか？')) {
                    return;
                }
                
                var $btn = $(this);
                $btn.prop('disabled', true);
                $('#wpap-repair-progress').show();
                $('#wpap-repair-results').hide();
                
                // まず全記事リストを取得
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'wpap_bulk_scan',
                        nonce: '<?php echo wp_create_nonce('wpap_bulk_scan'); ?>'
                    },
                    success: function(response) {
                        if (!response.success) {
                            alert('記事リストの取得に失敗しました。');
                            $btn.prop('disabled', false);
                            $('#wpap-repair-progress').hide();
                            return;
                        }
                        
                        var allPosts = response.data.posts.map(function(p) { return p.id; });
                        var total = allPosts.length;
                        var processed = 0;
                        var repaired = 0;
                        
                        function repairNext() {
                            if (allPosts.length === 0) {
                                $('#wpap-repair-bar').css('width', '100%');
                                $('#wpap-repair-text').text('完了！');
                                $('#wpap-repair-summary').text(processed + '件の記事を処理し、' + repaired + '件のショートコードを復元しました。');
                                $('#wpap-repair-results').show();
                                $btn.text('🔧 完了');
                                return;
                            }
                            
                            var postId = allPosts.shift();
                            var progress = Math.round((processed / total) * 100);
                            $('#wpap-repair-bar').css('width', progress + '%');
                            $('#wpap-repair-text').text('修復中... (' + processed + '/' + total + ')');
                            
                            $.ajax({
                                url: ajaxurl,
                                type: 'POST',
                                data: {
                                    action: 'wpap_force_repair',
                                    nonce: '<?php echo wp_create_nonce('wpap_force_repair'); ?>',
                                    post_id: postId
                                },
                                success: function(response) {
                                    processed++;
                                    if (response.success) {
                                        repaired += response.data.repaired_count || 0;
                                    }
                                    repairNext();
                                },
                                error: function() {
                                    processed++;
                                    repairNext();
                                }
                            });
                        }
                        
                        repairNext();
                    },
                    error: function() {
                        alert('通信エラーが発生しました。');
                        $btn.prop('disabled', false);
                        $('#wpap-repair-progress').hide();
                    }
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * Ajax: 全記事スキャン
     */
    public function ajax_bulk_scan() {
        check_ajax_referer('wpap_bulk_scan', 'nonce');
        
        if (!current_user_can('edit_posts')) {
            wp_send_json_error(__('権限がありません。', 'wpap-card-helper'));
        }
        
        $processor = new WPAP_Card_Helper_Shortcode_Processor();
        
        // 全ての投稿タイプを取得
        $post_types = get_post_types(array('public' => true), 'names');
        
        // 全ての投稿を取得（カスタム投稿タイプも含む）
        $posts = get_posts(array(
            'post_type' => array_values($post_types),
            'post_status' => array('publish', 'pending', 'draft', 'future', 'private', 'inherit'),
            'posts_per_page' => -1,
            'orderby' => 'ID',
            'order' => 'ASC',
        ));
        
        $results = array(
            'post_count' => 0,
            'unreplaced_count' => 0,
            'replaced_count' => 0,
            'posts' => array(),
            'posts_with_unreplaced' => array(),
            'posts_with_replaced' => array(),
        );
        
        foreach ($posts as $post) {
            $shortcodes = $processor->scan_shortcodes($post->post_content);
            
            if (empty($shortcodes)) {
                continue;
            }
            
            $unreplaced = count(array_filter($shortcodes, function($sc) {
                return !$sc['is_replaced'];
            }));
            
            $replaced = count(array_filter($shortcodes, function($sc) {
                return $sc['is_replaced'];
            }));
            
            $results['post_count']++;
            $results['unreplaced_count'] += $unreplaced;
            $results['replaced_count'] += $replaced;
            
            $results['posts'][] = array(
                'id' => $post->ID,
                'title' => $post->post_title,
                'edit_link' => get_edit_post_link($post->ID, 'raw'),
                'unreplaced' => $unreplaced,
                'replaced' => $replaced,
            );
            
            if ($unreplaced > 0) {
                $results['posts_with_unreplaced'][] = $post->ID;
            }
            if ($replaced > 0) {
                $results['posts_with_replaced'][] = $post->ID;
            }
        }
        
        wp_send_json_success($results);
    }
    
    /**
     * Ajax: 記事を置換
     */
    public function ajax_bulk_replace() {
        check_ajax_referer('wpap_bulk_replace', 'nonce');
        
        if (!current_user_can('edit_posts')) {
            wp_send_json_error(__('権限がありません。', 'wpap-card-helper'));
        }
        
        $post_id = intval($_POST['post_id']);
        if (empty($post_id)) {
            wp_send_json_error(__('投稿IDが指定されていません。', 'wpap-card-helper'));
        }
        
        $post = get_post($post_id);
        if (!$post) {
            wp_send_json_error(__('投稿が見つかりません。', 'wpap-card-helper'));
        }
        
        $processor = new WPAP_Card_Helper_Shortcode_Processor();
        $shortcodes = $processor->scan_shortcodes($post->post_content);
        
        $unreplaced = array_filter($shortcodes, function($sc) {
            return !$sc['is_replaced'];
        });
        
        if (empty($unreplaced)) {
            wp_send_json_success(array('replaced_count' => 0));
        }
        
        $content = $post->post_content;
        $replaced_count = 0;
        
        foreach ($unreplaced as $shortcode_info) {
            $asin = esc_attr($shortcode_info['asin']);
            $title = $this->escape_shortcode_attr($shortcode_info['title'] ?: '');

            $new_shortcode = '[wpap_card asin="' . $asin . '"';
            if (!empty($title)) {
                $new_shortcode .= ' title="' . $title . '"';
            }
            $new_shortcode .= ']';

            $content = $processor->replace_shortcode($content, $shortcode_info['shortcode'], $new_shortcode);
            $replaced_count++;
        }
        
        wp_update_post(array(
            'ID' => $post_id,
            'post_content' => $content,
        ));
        
        wp_send_json_success(array('replaced_count' => $replaced_count));
    }
    
    /**
     * Ajax: 記事を元に戻す
     */
    public function ajax_bulk_revert() {
        check_ajax_referer('wpap_bulk_revert', 'nonce');
        
        if (!current_user_can('edit_posts')) {
            wp_send_json_error(__('権限がありません。', 'wpap-card-helper'));
        }
        
        $post_id = intval($_POST['post_id']);
        if (empty($post_id)) {
            wp_send_json_error(__('投稿IDが指定されていません。', 'wpap-card-helper'));
        }
        
        $post = get_post($post_id);
        if (!$post) {
            wp_send_json_error(__('投稿が見つかりません。', 'wpap-card-helper'));
        }
        
        $processor = new WPAP_Card_Helper_Shortcode_Processor();
        
        // 元に戻す
        $content = $processor->revert_all($post->post_content);
        
        // 変更があったかチェック
        if ($content === $post->post_content) {
            wp_send_json_success(array('reverted_count' => 0));
        }
        
        // 復元されたショートコードをカウント
        $reverted_count = substr_count($content, '[wpap ') - substr_count($post->post_content, '[wpap ');
        if ($reverted_count < 0) {
            $reverted_count = 0;
        }
        // より正確なカウント: コメントが減った数
        $original_comments = preg_match_all('/<!-- wpap-original:/', $post->post_content, $m);
        $after_comments = preg_match_all('/<!-- wpap-original:/', $content, $m);
        $reverted_count = $original_comments - $after_comments;
        
        wp_update_post(array(
            'ID' => $post_id,
            'post_content' => $content,
        ));
        
        wp_send_json_success(array('reverted_count' => max(0, $reverted_count)));
    }
    
    /**
     * Ajax: 記事を強制修復・復元
     */
    public function ajax_force_repair() {
        check_ajax_referer('wpap_force_repair', 'nonce');
        
        if (!current_user_can('edit_posts')) {
            wp_send_json_error(__('権限がありません。', 'wpap-card-helper'));
        }
        
        $post_id = intval($_POST['post_id']);
        if (empty($post_id)) {
            wp_send_json_error(__('投稿IDが指定されていません。', 'wpap-card-helper'));
        }
        
        $post = get_post($post_id);
        if (!$post) {
            wp_send_json_error(__('投稿が見つかりません。', 'wpap-card-helper'));
        }
        
        $processor = new WPAP_Card_Helper_Shortcode_Processor();
        $content = $post->post_content;
        
        // 破損コンテンツを修復
        $content = $processor->repair_corrupted_content($content);
        
        // 元に戻す
        $content = $processor->revert_all($content);
        
        // 変更があったかチェック
        if ($content === $post->post_content) {
            wp_send_json_success(array('repaired_count' => 0));
        }
        
        // 復元されたショートコードをカウント
        $original_comments = preg_match_all('/<!-- wpap-original:/', $post->post_content, $m);
        $after_comments = preg_match_all('/<!-- wpap-original:/', $content, $m);
        $repaired_count = $original_comments - $after_comments;
        
        wp_update_post(array(
            'ID' => $post_id,
            'post_content' => $content,
        ));
        
        wp_send_json_success(array('repaired_count' => max(0, $repaired_count)));
    }
}
