<?php
/**
 * Amazon API連携クラス
 * PA-API 5.0 (AWS Signature V4) と Creators API (OAuth2) の両方をサポート
 */

if (!defined('ABSPATH')) {
    exit;
}

class WPAP_Card_Helper_Amazon_API {
    
    // ========================================
    // Creators API 設定
    // ========================================
    const CREATORS_API_BASE_URL = 'https://creatorsapi.amazon';
    const TOKEN_ENDPOINT = 'https://creatorsapi.auth.us-west-2.amazoncognito.com/oauth2/token';
    const CREATORS_API_VERSION = '2.3';
    
    // ========================================
    // 共通設定
    // ========================================
    const MARKETPLACE = 'www.amazon.co.jp';
    const TOKEN_TRANSIENT = 'wpap_replacer_token';
    
    /**
     * コンストラクタ
     */
    public function __construct() {
        // 必要に応じて初期化処理
    }
    
    /**
     * 商品情報の取得
     * Creators APIを使用
     * 
     * @param string|array $asins ASIN（単一または配列）
     * @return array|WP_Error
     */
    public function get_items($asins) {
        if (!is_array($asins)) {
            $asins = array($asins);
        }
        
        // 最大10件まで
        $asins = array_slice($asins, 0, 10);
        
        // Creators APIを使用
        return $this->get_items_creators($asins);
    }
    
    /**
     * Creators API で商品情報を取得（既存ロジック）
     */
    private function get_items_creators($asins) {
        $token = $this->get_access_token();
        if (is_wp_error($token)) {
            return $token;
        }
        
        $tracking_id = WPAP_Card_Helper_Settings::get_setting('tracking_id');
        if (empty($tracking_id)) {
            return new WP_Error('missing_tracking_id', __('トラッキングIDが設定されていません。', 'wpap-card-helper'));
        }
        
        // Creators API のリクエストボディ
        $request_body = array(
            'itemIds' => $asins,
            'itemIdType' => 'ASIN',
            'marketplace' => self::MARKETPLACE,
            'partnerTag' => $tracking_id,
            'resources' => array(
                'images.primary.large',
                'images.primary.medium',
                'itemInfo.title',
                'offersV2.listings.price',
                'offersV2.listings.availability',
            ),
        );
        
        $response = wp_remote_post(self::CREATORS_API_BASE_URL . '/catalog/v1/getItems', array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $token . ', Version ' . self::CREATORS_API_VERSION,
                'x-marketplace' => self::MARKETPLACE,
            ),
            'body' => json_encode($request_body),
            'timeout' => 30,
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if ($status_code === 401) {
            delete_transient(self::TOKEN_TRANSIENT);
            return new WP_Error('unauthorized', __('認証に失敗しました。', 'wpap-card-helper'));
        }

        if ($status_code !== 200) {
            $error_message = $body['errors'][0]['message'] ?? __('APIリクエストに失敗しました。', 'wpap-card-helper');
            return new WP_Error('api_error', $error_message);
        }
        
        return $this->parse_items_response($body);
    }
    
    /**
     * OAuth2トークンの取得（Creators API用）
     */
    private function get_access_token() {
        $cached_token = get_transient(self::TOKEN_TRANSIENT);
        if ($cached_token) {
            return $cached_token;
        }
        
        $credential_id = trim(WPAP_Card_Helper_Settings::get_setting('credential_id'));
        $credential_secret = trim(WPAP_Card_Helper_Settings::get_setting('credential_secret'));
        
        if (empty($credential_id) || empty($credential_secret)) {
            return new WP_Error('missing_credentials', __('APIクレデンシャルが設定されていません。', 'wpap-card-helper'));
        }
        
        $response = wp_remote_post(self::TOKEN_ENDPOINT, array(
            'headers' => array(
                'Content-Type' => 'application/x-www-form-urlencoded',
            ),
            'body' => http_build_query(array(
                'grant_type' => 'client_credentials',
                'client_id' => $credential_id,
                'client_secret' => $credential_secret,
                'scope' => 'creatorsapi/default',
            )),
            'timeout' => 30,
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($body['error'])) {
            return new WP_Error('token_error', $body['error_description'] ?? __('トークンエラー', 'wpap-card-helper'));
        }

        if (!isset($body['access_token'])) {
            return new WP_Error('token_error', __('アクセストークン不明', 'wpap-card-helper'));
        }
        
        $expires_in = isset($body['expires_in']) ? intval($body['expires_in']) - 60 : 3540;
        set_transient(self::TOKEN_TRANSIENT, $body['access_token'], $expires_in);
        
        return $body['access_token'];
    }
    
    /**
     * APIレスポンスのパース（両API対応）
     */
    private function parse_items_response($response) {
        $items = array();

        // トラッキングIDを取得
        $tracking_id = WPAP_Card_Helper_Settings::get_setting('tracking_id');

        // 形式の違いを吸収
        // Creators API
        $products = $response['itemsResult']['items'] ?? $response['items'] ?? array();

        foreach ($products as $item) {
            $asin = $item['asin'] ?? '';

            // デフォルトURLにトラッキングIDを付与
            $default_url = 'https://www.amazon.co.jp/dp/' . $asin;
            if (!empty($tracking_id)) {
                $default_url .= '?tag=' . $tracking_id;
            }

            $parsed_item = array(
                'asin' => $asin,
                'title' => '',
                'url' => $default_url,
                'image' => '',
                'price' => '',
                'availability' => '',
            );
            
            // Creators API マッピング (camelCase)
            if (isset($item['itemInfo']['title']['displayValue'])) {
                $parsed_item['title'] = $item['itemInfo']['title']['displayValue'];
            }
            if (isset($item['detailPageUrl'])) {
                $api_url = $item['detailPageUrl'];
                // APIから返されたURLにトラッキングIDがない場合は追加
                if (!empty($tracking_id) && strpos($api_url, 'tag=') === false) {
                    $separator = (strpos($api_url, '?') === false) ? '?' : '&';
                    $api_url .= $separator . 'tag=' . $tracking_id;
                }
                $parsed_item['url'] = $api_url;
            }
            if (isset($item['images']['primary']['large']['url'])) {
                $parsed_item['image'] = $item['images']['primary']['large']['url'];
            } elseif (isset($item['images']['primary']['medium']['url'])) {
                $parsed_item['image'] = $item['images']['primary']['medium']['url'];
            }
            if (isset($item['offersV2']['listings'][0]['price']['money']['displayAmount'])) {
                $parsed_item['price'] = $item['offersV2']['listings'][0]['price']['money']['displayAmount'];
            } elseif (isset($item['offersV2']['listings'][0]['price']['displayAmount'])) {
                $parsed_item['price'] = $item['offersV2']['listings'][0]['price']['displayAmount'];
            } elseif (isset($item['offers']['listings'][0]['price']['displayAmount'])) {
                $parsed_item['price'] = $item['offers']['listings'][0]['price']['displayAmount'];
            }
            if (isset($item['offersV2']['listings'][0]['availability']['message'])) {
                $parsed_item['availability'] = $item['offersV2']['listings'][0]['availability']['message'];
            }
            
            $items[$asin] = $parsed_item;
        }
        
        return $items;
    }
    
    /**
     * API接続テスト（Creators API）
     */
    public function test_connection() {
        $credential_id = WPAP_Card_Helper_Settings::get_setting('credential_id');
        $credential_secret = WPAP_Card_Helper_Settings::get_setting('credential_secret');
        $tracking_id = WPAP_Card_Helper_Settings::get_setting('tracking_id');
        
        if (empty($credential_id) || empty($credential_secret)) {
            return array('success' => false, 'message' => __('Creators API クレデンシャルが設定されていません。', 'wpap-card-helper'));
        }

        if (empty($tracking_id)) {
            return array('success' => false, 'message' => __('トラッキングIDが設定されていません。', 'wpap-card-helper'));
        }

        // Creators API の接続テスト
        $token = $this->get_access_token();
        if (is_wp_error($token)) {
            return array('success' => false, 'message' => $token->get_error_message());
        }

        return array('success' => true, 'message' => __('Creators API 接続テスト成功！', 'wpap-card-helper'));
    }
}
