/**
 * WPAP Replacer - クラシックエディタ用JavaScript
 */

(function ($) {
    'use strict';

    var WPAPReplacer = {

        /**
         * 検出されたショートコード
         */
        detectedShortcodes: [],

        /**
         * 初期化
         */
        init: function () {
            this.bindEvents();
            this.checkInitialState();
        },

        /**
         * イベントのバインド
         */
        bindEvents: function () {
            $('#wpap-scan-btn').on('click', this.scanShortcodes.bind(this));
            $('#wpap-replace-btn').on('click', this.replaceShortcodes.bind(this));
            $('#wpap-revert-btn').on('click', this.revertShortcodes.bind(this));
        },

        /**
         * 初期状態のチェック
         */
        checkInitialState: function () {
            // ページ読み込み時に自動スキャン（オプション）
            // this.scanShortcodes();
        },

        /**
         * ショートコードのスキャン
         */
        scanShortcodes: function () {
            var self = this;
            var $btn = $('#wpap-scan-btn');
            var $results = $('#wpap-replacer-results');

            this.setButtonLoading($btn, true, '検出中...');
            this.showStatus('ショートコードを検出しています...', 'info');

            $.ajax({
                url: wpapCardHelperData.restUrl + 'scan',
                method: 'POST',
                beforeSend: function (xhr) {
                    xhr.setRequestHeader('X-WP-Nonce', wpapCardHelperData.nonce);
                },
                data: {
                    post_id: wpapCardHelperData.postId
                },
                success: function (response) {
                    if (response.success) {
                        self.detectedShortcodes = response.data.shortcodes;
                        self.renderResults(response.data);
                        self.updateButtonStates(response.data);

                        if (response.data.total === 0) {
                            self.showStatus('WPAPショートコードは見つかりませんでした。', 'warning');
                        } else {
                            self.showStatus(
                                response.data.total + '件のショートコードを検出しました。' +
                                '（未置換: ' + response.data.unreplaced_count + '件、' +
                                '置換済み: ' + response.data.replaced_count + '件）',
                                'success'
                            );
                        }
                    } else {
                        self.showStatus('検出に失敗しました。', 'error');
                    }
                },
                error: function (xhr) {
                    var message = xhr.responseJSON ? xhr.responseJSON.message : 'エラーが発生しました';
                    self.showStatus(message, 'error');
                },
                complete: function () {
                    self.setButtonLoading($btn, false, '🔍 ショートコードを検出');
                }
            });
        },

        /**
         * 検出結果の表示
         */
        renderResults: function (data) {
            var $results = $('#wpap-replacer-results');
            $results.empty();

            if (data.shortcodes.length === 0) {
                $results.html('<p style="color: #666; text-align: center;">ショートコードが見つかりません</p>');
                return;
            }

            var html = '<ul style="margin: 0; padding: 0; list-style: none;">';

            data.shortcodes.forEach(function (sc) {
                var status = sc.is_replaced
                    ? '<span style="color: green;">✓ 置換済み</span>'
                    : '<span style="color: orange;">○ 未置換</span>';

                var title = sc.title || sc.asin;
                if (title.length > 30) {
                    title = title.substring(0, 30) + '...';
                }

                html += '<li style="padding: 5px; border-bottom: 1px solid #eee; font-size: 12px;">';
                html += '<strong>' + self.escapeHtml(title) + '</strong><br>';
                html += '<span style="color: #666;">ASIN: ' + sc.asin + '</span><br>';
                html += status;
                html += '</li>';
            });

            html += '</ul>';
            $results.html(html);
        },

        /**
         * ボタン状態の更新
         */
        updateButtonStates: function (data) {
            $('#wpap-replace-btn').prop('disabled', data.unreplaced_count === 0);
            $('#wpap-revert-btn').prop('disabled', data.replaced_count === 0);
        },

        /**
         * ショートコードの置換
         */
        replaceShortcodes: function () {
            var self = this;
            var $btn = $('#wpap-replace-btn');

            if (!confirm('検出されたすべてのショートコードをAmazonアフィリエイトカードに置換しますか？')) {
                return;
            }

            this.setButtonLoading($btn, true, '置換中...');
            this.showProgress(0, '商品情報を取得しています...');

            $.ajax({
                url: wpapCardHelperData.restUrl + 'replace',
                method: 'POST',
                beforeSend: function (xhr) {
                    xhr.setRequestHeader('X-WP-Nonce', wpapCardHelperData.nonce);
                },
                data: {
                    post_id: wpapCardHelperData.postId
                },
                success: function (response) {
                    self.hideProgress();

                    if (response.success) {
                        self.showStatus(response.message, 'success');

                        // エディタのコンテンツを更新
                        self.refreshEditorContent();

                        // 再スキャン
                        setTimeout(function () {
                            self.scanShortcodes();
                        }, 500);
                    } else {
                        self.showStatus(response.message || '置換に失敗しました。', 'error');
                    }
                },
                error: function (xhr) {
                    self.hideProgress();
                    var message = xhr.responseJSON ? xhr.responseJSON.message : 'エラーが発生しました';
                    self.showStatus(message, 'error');
                },
                complete: function () {
                    self.setButtonLoading($btn, false, '✨ すべて置換');
                }
            });
        },

        /**
         * 元に戻す
         */
        revertShortcodes: function () {
            var self = this;
            var $btn = $('#wpap-revert-btn');

            if (!confirm('すべてのアフィリエイトカードを元のWPAPショートコードに戻しますか？')) {
                return;
            }

            this.setButtonLoading($btn, true, '復元中...');

            $.ajax({
                url: wpapCardHelperData.restUrl + 'revert',
                method: 'POST',
                beforeSend: function (xhr) {
                    xhr.setRequestHeader('X-WP-Nonce', wpapCardHelperData.nonce);
                },
                data: {
                    post_id: wpapCardHelperData.postId
                },
                success: function (response) {
                    if (response.success) {
                        self.showStatus(response.message, 'success');

                        // エディタのコンテンツを更新
                        self.refreshEditorContent();

                        // 再スキャン
                        setTimeout(function () {
                            self.scanShortcodes();
                        }, 500);
                    } else {
                        self.showStatus(response.message || '復元に失敗しました。', 'error');
                    }
                },
                error: function (xhr) {
                    var message = xhr.responseJSON ? xhr.responseJSON.message : 'エラーが発生しました';
                    self.showStatus(message, 'error');
                },
                complete: function () {
                    self.setButtonLoading($btn, false, '↩️ 元に戻す');
                }
            });
        },

        /**
         * エディタのコンテンツを更新
         */
        refreshEditorContent: function () {
            // ページをリロードしてコンテンツを更新
            // より良い方法：AJAXで投稿を取得してエディタを更新
            location.reload();
        },

        /**
         * ステータス表示
         */
        showStatus: function (message, type) {
            var $status = $('#wpap-replacer-status');
            var $text = $('#wpap-status-text');

            var bgColor = {
                'info': '#e7f3fe',
                'success': '#d4edda',
                'warning': '#fff3cd',
                'error': '#f8d7da'
            };

            var textColor = {
                'info': '#0c5460',
                'success': '#155724',
                'warning': '#856404',
                'error': '#721c24'
            };

            $status.css({
                'background': bgColor[type] || bgColor.info,
                'color': textColor[type] || textColor.info
            }).show();

            $text.text(message);
        },

        /**
         * プログレス表示
         */
        showProgress: function (percent, text) {
            var $progress = $('#wpap-replacer-progress');
            var $bar = $('#wpap-progress-bar');
            var $text = $('#wpap-progress-text');

            $progress.show();
            $bar.css('width', percent + '%');
            $text.text(text);
        },

        /**
         * プログレス非表示
         */
        hideProgress: function () {
            $('#wpap-replacer-progress').hide();
        },

        /**
         * ボタンのローディング状態
         */
        setButtonLoading: function ($btn, loading, text) {
            $btn.prop('disabled', loading).text(text);
        },

        /**
         * HTMLエスケープ
         */
        escapeHtml: function (text) {
            var div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
    };

    // 自己参照用
    var self = WPAPReplacer;

    // DOM Ready
    $(document).ready(function () {
        WPAPReplacer.init();
    });

})(jQuery);
